/* LoongArch opcode support.
   Copyright (C) 2021-2025 Free Software Foundation, Inc.
   Contributed by Loongson Ltd.

   This file is part of the GNU opcodes library.

   This library is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   It is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
   License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING3.  If not,
   see <http://www.gnu.org/licenses/>.  */

#include <stddef.h>
#include "opcode/loongarch.h"
#include "libiberty.h"

struct loongarch_ASEs_option LARCH_opts =
{
  .relax = 1,
  .thin_add_sub = 0
};

size_t
loongarch_insn_length (insn_t insn ATTRIBUTE_UNUSED)
{
  return 4;
}

const char *const loongarch_r_normal_name[32] =
{
  "$r0",  "$r1",  "$r2",  "$r3",  "$r4",  "$r5",  "$r6",  "$r7",
  "$r8",  "$r9",  "$r10", "$r11", "$r12", "$r13", "$r14", "$r15",
  "$r16", "$r17", "$r18", "$r19", "$r20", "$r21", "$r22", "$r23",
  "$r24", "$r25", "$r26", "$r27", "$r28", "$r29", "$r30", "$r31",
};

const char *const loongarch_r_alias[32] =
{
  "$zero", "$ra", "$tp", "$sp", "$a0", "$a1", "$a2", "$a3",
  "$a4",   "$a5", "$a6", "$a7", "$t0", "$t1", "$t2", "$t3",
  "$t4",   "$t5", "$t6", "$t7", "$t8", "$r21","$fp", "$s0",
  "$s1",   "$s2", "$s3", "$s4", "$s5", "$s6", "$s7", "$s8",
};

/* Add support for $s9.  */
const char *const loongarch_r_alias_1[32] =
{
  "", "", "", "", "", "", "", "",
  "", "", "", "", "", "", "", "",
  "", "", "", "", "", "", "$s9", "",
  "", "", "", "", "", "", "", "",
};

const char *const loongarch_r_alias_deprecated[32] =
{
  "", "", "", "", "$v0", "$v1", "", "", "", "", "", "", "", "", "", "",
  "", "", "", "", "",    "$x",  "", "", "", "", "", "", "", "", "", "",
};

const char *const loongarch_f_normal_name[32] =
{
  "$f0",  "$f1",  "$f2",  "$f3",  "$f4",  "$f5",  "$f6",  "$f7",
  "$f8",  "$f9",  "$f10", "$f11", "$f12", "$f13", "$f14", "$f15",
  "$f16", "$f17", "$f18", "$f19", "$f20", "$f21", "$f22", "$f23",
  "$f24", "$f25", "$f26", "$f27", "$f28", "$f29", "$f30", "$f31",
};

const char *const loongarch_f_alias[32] =
{
  "$fa0", "$fa1", "$fa2",  "$fa3",  "$fa4",  "$fa5",  "$fa6",  "$fa7",
  "$ft0", "$ft1", "$ft2",  "$ft3",  "$ft4",  "$ft5",  "$ft6",  "$ft7",
  "$ft8", "$ft9", "$ft10", "$ft11", "$ft12", "$ft13", "$ft14", "$ft15",
  "$fs0", "$fs1", "$fs2",  "$fs3",  "$fs4",  "$fs5",  "$fs6",  "$fs7",
};

const char *const loongarch_f_alias_deprecated[32] =
{
  "$fv0", "$fv1", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
  "",     "",     "", "", "", "", "", "", "", "", "", "", "", "", "", "",
};

const char *const loongarch_fc_normal_name[4] =
{
  "$fcsr0", "$fcsr1", "$fcsr2", "$fcsr3",
};

const char *const loongarch_fc_numeric_name[4] =
{
  "$r0", "$r1", "$r2", "$r3",
};

const char *const loongarch_c_normal_name[8] =
{
  "$fcc0", "$fcc1", "$fcc2", "$fcc3", "$fcc4", "$fcc5", "$fcc6", "$fcc7",
};

const char *const loongarch_cr_normal_name[4] =
{
  "$scr0",
  "$scr1",
  "$scr2",
  "$scr3",
};

const char *const loongarch_v_normal_name[32] =
{
  "$vr0",  "$vr1",  "$vr2",  "$vr3",  "$vr4",  "$vr5",  "$vr6",  "$vr7",
  "$vr8",  "$vr9",  "$vr10", "$vr11", "$vr12", "$vr13", "$vr14", "$vr15",
  "$vr16", "$vr17", "$vr18", "$vr19", "$vr20", "$vr21", "$vr22", "$vr23",
  "$vr24", "$vr25", "$vr26", "$vr27", "$vr28", "$vr29", "$vr30", "$vr31",
};

const char *const loongarch_x_normal_name[32] =
{
  "$xr0",  "$xr1",  "$xr2",  "$xr3",  "$xr4",  "$xr5",  "$xr6",  "$xr7",
  "$xr8",  "$xr9",  "$xr10", "$xr11", "$xr12", "$xr13", "$xr14", "$xr15",
  "$xr16", "$xr17", "$xr18", "$xr19", "$xr20", "$xr21", "$xr22", "$xr23",
  "$xr24", "$xr25", "$xr26", "$xr27", "$xr28", "$xr29", "$xr30", "$xr31",
};

const char *const loongarch_r_cfi_name[32] =
{
  "r0",  "r1",  "r2",  "r3",  "r4",  "r5",  "r6",  "r7",
  "r8",  "r9",  "r10", "r11", "r12", "r13", "r14", "r15",
  "r16", "r17", "r18", "r19", "r20", "r21", "r22", "r23",
  "r24", "r25", "r26", "r27", "r28", "r29", "r30", "r31",
};

const char *const loongarch_r_cfi_name_alias[32] =
{
  "zero", "ra", "tp", "sp", "a0", "a1", "a2", "a3",
  "a4",   "a5", "a6", "a7", "t0", "t1", "t2", "t3",
  "t4",   "t5", "t6", "t7", "t8", "r21","fp", "s0",
  "s1",   "s2", "s3", "s4", "s5", "s6", "s7", "s8",
};

const char *const loongarch_f_cfi_name[32] =
{
  "f0",  "f1",  "f2",  "f3",  "f4",  "f5",  "f6",  "f7",
  "f8",  "f9",  "f10", "f11", "f12", "f13", "f14", "f15",
  "f16", "f17", "f18", "f19", "f20", "f21", "f22", "f23",
  "f24", "f25", "f26", "f27", "f28", "f29", "f30", "f31",
};

const char *const loongarch_f_cfi_name_alias[32] =
{
  "fa0", "fa1", "fa2",  "fa3",  "fa4",  "fa5",  "fa6",  "fa7",
  "ft0", "ft1", "ft2",  "ft3",  "ft4",  "ft5",  "ft6",  "ft7",
  "ft8", "ft9", "ft10", "ft11", "ft12", "ft13", "ft14", "ft15",
  "fs0", "fs1", "fs2",  "fs3",  "fs4",  "fs5",  "fs6",  "fs7",
};

/* Can not use xx_pa for abs.  */

/* For LoongArch32 abs.  */
#define INSN_LA_ABS32		    \
  "lu12i.w %1,%%abs_hi20(%2);"	    \
  "ori %1,%1,%%abs_lo12(%2);",	    \
  &LARCH_opts.ase_ilp32,	    \
  &LARCH_opts.ase_lp64
#define INSN_LA_ABS64			\
  "lu12i.w %1,%%abs_hi20(%2);"		\
  "ori %1,%1,%%abs_lo12(%2);"		\
  "lu32i.d %1,%%abs64_lo20(%2);"	\
  "lu52i.d %1,%1,%%abs64_hi12(%2);",	\
  &LARCH_opts.ase_lp64, 0

#define INSN_LA_PCREL32		    \
  "pcalau12i %1,%%pc_hi20(%2);"	    \
  "addi.w %1,%1,%%pc_lo12(%2);",    \
  &LARCH_opts.ase_ilp32,	    \
  &LARCH_opts.ase_lp64
#define INSN_LA_PCREL64		    \
  "pcalau12i %1,%%pc_hi20(%2);"	    \
  "addi.d %1,%1,%%pc_lo12(%2);",    \
  &LARCH_opts.ase_lp64, 0
#define INSN_LA_PCREL64_LARGE	    \
  "pcalau12i %1,%%pc_hi20(%3);"	    \
  "addi.d %2,$r0,%%pc_lo12(%3);"    \
  "lu32i.d %2,%%pc64_lo20(%3);"	    \
  "lu52i.d %2,%2,%%pc64_hi12(%3);"  \
  "add.d %1,%1,%2;",		    \
  &LARCH_opts.ase_lp64, 0

#define INSN_LA_GOT32		    \
  "pcalau12i %1,%%got_pc_hi20(%2);" \
  "ld.w %1,%1,%%got_pc_lo12(%2);",  \
  &LARCH_opts.ase_ilp32,	    \
  &LARCH_opts.ase_lp64
/* got32 abs.  */
#define INSN_LA_GOT32_ABS	    \
  "lu12i.w %1,%%got_hi20(%2);"	    \
  "ori %1,%1,%%got_lo12(%2);"	    \
  "ld.w %1,%1,0;",		    \
  &LARCH_opts.ase_gabs,		    \
  &LARCH_opts.ase_lp64
#define INSN_LA_GOT64		    \
  "pcalau12i %1,%%got_pc_hi20(%2);" \
  "ld.d %1,%1,%%got_pc_lo12(%2);",  \
  &LARCH_opts.ase_lp64, 0
/* got64 abs.  */
#define INSN_LA_GOT64_LARGE_ABS	    \
  "lu12i.w %1,%%got_hi20(%2);"	    \
  "ori %1,%1,%%got_lo12(%2);"	    \
  "lu32i.d %1,%%got64_lo20(%2);"    \
  "lu52i.d %1,%1,%%got64_hi12(%2);" \
  "ld.d %1,%1,0",		    \
  &LARCH_opts.ase_gabs,		    \
  &LARCH_opts.ase_gpcr
/* got64 pic.  */
#define INSN_LA_GOT64_LARGE_PCREL     \
  "pcalau12i %1,%%got_pc_hi20(%3);"   \
  "addi.d %2,$r0,%%got_pc_lo12(%3);"  \
  "lu32i.d %2,%%got64_pc_lo20(%3);"   \
  "lu52i.d %2,%2,%%got64_pc_hi12(%3);"\
  "ldx.d %1,%1,%2;",		      \
  &LARCH_opts.ase_lp64,		      \
  &LARCH_opts.ase_gabs

/* For LoongArch32/64 cmode=normal.  */
#define INSN_LA_TLS_LE		    \
  "lu12i.w %1,%%le_hi20(%2);"	    \
  "ori %1,%1,%%le_lo12(%2);",	    \
  &LARCH_opts.ase_ilp32, 0

/* For LoongArch64 cmode=large.  */
#define INSN_LA_TLS_LE64_LARGE	    \
  "lu12i.w %1,%%le_hi20(%2);"	    \
  "ori %1,%1,%%le_lo12(%2);"	    \
  "lu32i.d %1,%%le64_lo20(%2);"	    \
  "lu52i.d %1,%1,%%le64_hi12(%2);", \
  &LARCH_opts.ase_lp64, 0

#define INSN_LA_TLS_IE32	    \
  "pcalau12i %1,%%ie_pc_hi20(%2);"  \
  "ld.w %1,%1,%%ie_pc_lo12(%2);",   \
  &LARCH_opts.ase_ilp32,	    \
  &LARCH_opts.ase_lp64
/* For ie32 abs.  */
#define INSN_LA_TLS_IE32_ABS	    \
  "lu12i.w %1,%%ie_hi20(%2);"	    \
  "ori %1,%1,%%ie_lo12(%2);"	    \
  "ld.w %1,%1,0",		    \
  &LARCH_opts.ase_gabs,		    \
  &LARCH_opts.ase_lp64
#define INSN_LA_TLS_IE64	    \
  "pcalau12i %1,%%ie_pc_hi20(%2);"  \
  "ld.d %1,%1,%%ie_pc_lo12(%2);",   \
  &LARCH_opts.ase_lp64, 0
/* For ie64 pic.  */
#define INSN_LA_TLS_IE64_LARGE_PCREL  \
  "pcalau12i %1,%%ie_pc_hi20(%3);"    \
  "addi.d %2,$r0,%%ie_pc_lo12(%3);"   \
  "lu32i.d %2,%%ie64_pc_lo20(%3);"    \
  "lu52i.d %2,%2,%%ie64_pc_hi12(%3);" \
  "ldx.d %1,%1,%2;",		      \
  &LARCH_opts.ase_lp64,		      \
  &LARCH_opts.ase_gabs
/* For ie64 abs.  */
#define INSN_LA_TLS_IE64_LARGE_ABS  \
  "lu12i.w %1,%%ie_hi20(%2);"	    \
  "ori %1,%1,%%ie_lo12(%2);"	    \
  "lu32i.d %1,%%ie64_lo20(%2);"	    \
  "lu52i.d %1,%1,%%ie64_hi12(%2);"  \
  "ld.d %1,%1,0",		    \
  &LARCH_opts.ase_gabs,		    \
  &LARCH_opts.ase_gpcr

/* For LoongArch32/64 cmode=normal.  */
#define INSN_LA_TLS_LD32	      \
  "pcalau12i %1,%%ld_pc_hi20(%2);"    \
  "addi.w %1,%1,%%got_pc_lo12(%2);",  \
  &LARCH_opts.ase_ilp32,	      \
  &LARCH_opts.ase_lp64
#define INSN_LA_TLS_LD32_ABS	      \
  "lu12i.w %1,%%ld_hi20(%2);"	      \
  "ori %1,%1,%%got_lo12(%2);",	      \
  &LARCH_opts.ase_gabs,		      \
  &LARCH_opts.ase_lp64
#define INSN_LA_TLS_LD64	      \
  "pcalau12i %1,%%ld_pc_hi20(%2);"    \
  "addi.d %1,%1,%%got_pc_lo12(%2);",  \
  &LARCH_opts.ase_lp64, 0
#define INSN_LA_TLS_LD64_LARGE_PCREL  \
  "pcalau12i %1,%%ld_pc_hi20(%3);"    \
  "addi.d %2,$r0,%%got_pc_lo12(%3);"  \
  "lu32i.d %2,%%got64_pc_lo20(%3);"   \
  "lu52i.d %2,%2,%%got64_pc_hi12(%3);"\
  "add.d %1,%1,%2;",		      \
  &LARCH_opts.ase_lp64,		      \
  &LARCH_opts.ase_gabs
#define INSN_LA_TLS_LD64_LARGE_ABS    \
  "lu12i.w %1,%%ld_hi20(%2);"	      \
  "ori %1,%1,%%got_lo12(%2);"	      \
  "lu32i.d %1,%%got64_lo20(%2);"      \
  "lu52i.d %1,%1,%%got64_hi12(%2);",  \
  &LARCH_opts.ase_gabs,		      \
  &LARCH_opts.ase_gpcr

#define INSN_LA_TLS_GD32	      \
  "pcalau12i %1,%%gd_pc_hi20(%2);"    \
  "addi.w %1,%1,%%got_pc_lo12(%2);",  \
  &LARCH_opts.ase_ilp32,	      \
  &LARCH_opts.ase_lp64
#define INSN_LA_TLS_GD32_ABS	      \
  "lu12i.w %1,%%gd_hi20(%2);"	      \
  "ori %1,%1,%%got_lo12(%2);",	      \
  &LARCH_opts.ase_gabs,		      \
  &LARCH_opts.ase_lp64
#define INSN_LA_TLS_GD64	      \
  "pcalau12i %1,%%gd_pc_hi20(%2);"    \
  "addi.d %1,%1,%%got_pc_lo12(%2);",  \
  &LARCH_opts.ase_lp64, 0
#define INSN_LA_TLS_GD64_LARGE_PCREL  \
  "pcalau12i %1,%%gd_pc_hi20(%3);"    \
  "addi.d %2,$r0,%%got_pc_lo12(%3);"  \
  "lu32i.d %2,%%got64_pc_lo20(%3);"   \
  "lu52i.d %2,%2,%%got64_pc_hi12(%3);"\
  "add.d %1,%1,%2;",		      \
  &LARCH_opts.ase_lp64,		      \
  &LARCH_opts.ase_gabs
#define INSN_LA_TLS_GD64_LARGE_ABS    \
  "lu12i.w %1,%%gd_hi20(%2);"	      \
  "ori %1,%1,%%got_lo12(%2);"	      \
  "lu32i.d %1,%%got64_lo20(%2);"      \
  "lu52i.d %1,%1,%%got64_hi12(%2);",  \
  &LARCH_opts.ase_gabs,		      \
  &LARCH_opts.ase_gpcr

#define INSN_LA_CALL		      \
  "pcaddu18i $ra,%%call36(%1);"    \
  "jirl $ra,$ra,0;",    \
  0, 0

#define INSN_LA_TAIL		      \
  "pcaddu18i %1,%%call36(%2);"   \
  "jirl $zero,%1,0;",   \
  0, 0

/* For TLS_DESC32 pcrel.  */
#define INSN_LA_TLS_DESC32		\
  "pcalau12i $r4,%%desc_pc_hi20(%2);"	\
  "addi.w $r4,$r4,%%desc_pc_lo12(%2);"	\
  "ld.w $r1,$r4,%%desc_ld(%2);"		\
  "jirl $r1,$r1,%%desc_call(%2);",	\
  &LARCH_opts.ase_ilp32,		\
  &LARCH_opts.ase_lp64

/* For TLS_DESC32 abs.  */
#define INSN_LA_TLS_DESC32_ABS		\
  "lu12i.w $r4,%%desc_hi20(%2);"	\
  "ori $r4,$r4,%%desc_lo12(%2);"	\
  "ld.w $r1,$r4,%%desc_ld(%2);"		\
  "jirl $r1,$r1,%%desc_call(%2);",	\
  &LARCH_opts.ase_gabs,			\
  &LARCH_opts.ase_lp64

/* For TLS_DESC64 pcrel.  */
#define INSN_LA_TLS_DESC64		\
  "pcalau12i $r4,%%desc_pc_hi20(%2);"	\
  "addi.d $r4,$r4,%%desc_pc_lo12(%2);"	\
  "ld.d $r1,$r4,%%desc_ld(%2);"		\
  "jirl $r1,$r1,%%desc_call(%2);",	\
  &LARCH_opts.ase_lp64, 0

/* For TLS_DESC64 large pcrel.  */
#define INSN_LA_TLS_DESC64_LARGE_PCREL	\
  "pcalau12i $r4,%%desc_pc_hi20(%3);"	\
  "addi.d %2,$r0,%%desc_pc_lo12(%3);"	\
  "lu32i.d %2,%%desc64_pc_lo20(%3);"	\
  "lu52i.d %2,%2,%%desc64_pc_hi12(%3);"	\
  "add.d $r4,$r4,%2;"			\
  "ld.d $r1,$r4,%%desc_ld(%3);"		\
  "jirl $r1,$r1,%%desc_call(%3);",	\
  &LARCH_opts.ase_lp64,			\
  &LARCH_opts.ase_gabs

/* For TLS_DESC64 large abs.  */
#define INSN_LA_TLS_DESC64_LARGE_ABS	\
  "lu12i.w $r4,%%desc_hi20(%2);"	\
  "ori $r4,$r4,%%desc_lo12(%2);"	\
  "lu32i.d $r4,%%desc64_lo20(%2);"	\
  "lu52i.d $r4,$r4,%%desc64_hi12(%2);"	\
  "ld.d $r1,$r4,%%desc_ld(%2);"		\
  "jirl $r1,$r1,%%desc_call(%2);",	\
  &LARCH_opts.ase_gabs,			\
  &LARCH_opts.ase_gpcr

static struct loongarch_opcode loongarch_macro_opcodes[] =
{
  /* match,    mask,	   name, format, macro, include, exclude, pinfo.  */
  { 0, 0, "li.w", "r,sc", "%f",	0, 0, 0 },
  { 0, 0, "li.d", "r,sc", "%f",	0, 0, 0 },

  { 0, 0, "la",		"r,la",	  "la.global %1,%2",	0,			0, 0 },
  { 0, 0, "la.global",	"r,la",	  "la.pcrel %1,%2",	&LARCH_opts.ase_gpcr,	0, 0 },
  { 0, 0, "la.global",	"r,r,la", "la.pcrel %1,%2,%3",	&LARCH_opts.ase_gpcr,	0, 0 },
  { 0, 0, "la.global",	"r,la",	  "la.abs %1,%2",	&LARCH_opts.ase_gabs,	0, 0 },
  { 0, 0, "la.global",	"r,r,la", "la.abs %1,%3",	&LARCH_opts.ase_gabs,	0, 0 },
  { 0, 0, "la.global",	"r,la",	  "la.got %1,%2",	0,			0, 0 },
  { 0, 0, "la.global",	"r,r,la", "la.got %1,%2,%3",	&LARCH_opts.ase_lp64,	0, 0 },

  { 0, 0, "la.local",	"r,la",	  "la.abs %1,%2",	&LARCH_opts.ase_labs,	0, 0 },
  { 0, 0, "la.local",	"r,r,la", "la.abs %1,%3",	&LARCH_opts.ase_labs,	0, 0 },
  { 0, 0, "la.local",	"r,la",	  "la.pcrel %1,%2",	0,			0, 0 },
  { 0, 0, "la.local",	"r,r,la", "la.pcrel %1,%2,%3",	&LARCH_opts.ase_lp64,	0, 0 },

  { 0, 0, "la.abs",	"r,la",	  INSN_LA_ABS32,		0 },
  { 0, 0, "la.abs",	"r,la",	  INSN_LA_ABS64,		0 },
  { 0, 0, "la.pcrel",	"r,la",	  INSN_LA_PCREL32,		0 },
  { 0, 0, "la.pcrel",	"r,la",	  INSN_LA_PCREL64,		0 },
  { 0, 0, "la.pcrel",	"r,r,la", INSN_LA_PCREL64_LARGE,	0 },
  { 0, 0, "la.got",	"r,la",	  INSN_LA_GOT32_ABS,		0 },
  { 0, 0, "la.got",	"r,la",	  INSN_LA_GOT32,		0 },
  { 0, 0, "la.got",	"r,la",	  INSN_LA_GOT64_LARGE_ABS,	0 },
  { 0, 0, "la.got",	"r,la",	  INSN_LA_GOT64,		0 },
  { 0, 0, "la.got",	"r,r,la", INSN_LA_GOT64_LARGE_PCREL,	0 },
  { 0, 0, "la.tls.le",	"r,l",	  INSN_LA_TLS_LE,		0 },
  { 0, 0, "la.tls.le",	"r,l",	  INSN_LA_TLS_LE64_LARGE,	0 },
  { 0, 0, "la.tls.ie",	"r,l",	  INSN_LA_TLS_IE32_ABS,		0 },
  { 0, 0, "la.tls.ie",	"r,l",	  INSN_LA_TLS_IE32,		0 },
  { 0, 0, "la.tls.ie",	"r,l",	  INSN_LA_TLS_IE64_LARGE_ABS,	0 },
  { 0, 0, "la.tls.ie",	"r,l",	  INSN_LA_TLS_IE64,		0 },
  { 0, 0, "la.tls.ie",	"r,r,l",  INSN_LA_TLS_IE64_LARGE_PCREL,	0 },
  { 0, 0, "la.tls.ld",	"r,l",	  INSN_LA_TLS_LD32_ABS,		0 },
  { 0, 0, "la.tls.ld",	"r,l",	  INSN_LA_TLS_LD32,		0 },
  { 0, 0, "la.tls.ld",	"r,l",	  INSN_LA_TLS_LD64_LARGE_ABS,	0 },
  { 0, 0, "la.tls.ld",	"r,l",	  INSN_LA_TLS_LD64,		0 },
  { 0, 0, "la.tls.ld",	"r,r,l",  INSN_LA_TLS_LD64_LARGE_PCREL,	0 },
  { 0, 0, "la.tls.gd",	"r,l",	  INSN_LA_TLS_GD32_ABS,		0 },
  { 0, 0, "la.tls.gd",	"r,l",	  INSN_LA_TLS_GD32,		0 },
  { 0, 0, "la.tls.gd",	"r,l",	  INSN_LA_TLS_GD64_LARGE_ABS,	0 },
  { 0, 0, "la.tls.gd",	"r,l",	  INSN_LA_TLS_GD64,		0 },
  { 0, 0, "la.tls.gd",	"r,r,l",  INSN_LA_TLS_GD64_LARGE_PCREL,	0 },
  { 0, 0, "call36",	"la",	  INSN_LA_CALL,			0 },
  { 0, 0, "tail36",	"r,la",	  INSN_LA_TAIL,			0 },
  { 0, 0, "pcaddi",	"r,la",	  "pcaddi %1, %%pcrel_20(%2)",	&LARCH_opts.ase_ilp32, 0, 0 },
  { 0, 0, "la.tls.desc", "r,l",	  INSN_LA_TLS_DESC32_ABS,	0 },
  { 0, 0, "la.tls.desc", "r,l",	  INSN_LA_TLS_DESC32,		0 },
  { 0, 0, "la.tls.desc", "r,l",	  INSN_LA_TLS_DESC64_LARGE_ABS,	0 },
  { 0, 0, "la.tls.desc", "r,l",	  INSN_LA_TLS_DESC64,		0 },
  { 0, 0, "la.tls.desc", "r,r,l", INSN_LA_TLS_DESC64_LARGE_PCREL,0 },
  { 0, 0, "ud",		 "u",	  "amswap.w $r%1,$r1,$r%1",	0, 0, 0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_alias_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,	include, exclude, pinfo.  */
  { 0x00006000, 0xffffffe0,	"rdcntvl.w",	"r0:5",				0,	0, 0, INSN_DIS_ALIAS }, /* rdtimel.w rd, zero */
  { 0x00006000, 0xfffffc1f,	"rdcntid.w",	"r5:5",				0,	0, 0, INSN_DIS_ALIAS }, /* rdtimel.w zero, rj */
  { 0x00006400, 0xffffffe0,	"rdcntvh.w",	"r0:5",				0,	0, 0, INSN_DIS_ALIAS }, /* rdtimeh.w rd, zero */
  { 0x00150000,	0xfffffc00,	"move",		"r0:5,r5:5",			0,	0, 0, INSN_DIS_ALIAS }, /* or rd, rj, zero */
  { 0x02800000, 0xffc003e0,	"li.w",		"r0:5,s10:12",			0,	0, 0, INSN_DIS_ALIAS }, /* addi.w rd, zero, simm */
  { 0x02c00000, 0xffc003e0,	"li.d",		"r0:5,s10:12",			0,	0, 0, INSN_DIS_ALIAS }, /* addi.d rd, zero, simm */
  { 0x03400000,	0xffffffff,	"nop",		"",				0,	0, 0, INSN_DIS_ALIAS }, /* andi zero, zero, 0 */
  { 0x03800000, 0xffc003e0,	"li.w",		"r0:5,u10:12",			0,	0, 0, INSN_DIS_ALIAS }, /* ori rd, zero, uimm */
  /* ret must come before jr because it is more specific.  */
  { 0x4c000020,	0xffffffff,	"ret",		"",				0,	0, 0, INSN_DIS_ALIAS }, /* jirl zero, ra, 0 */
  { 0x4c000000,	0xfffffc1f,	"jr",		"r5:5",				0,	0, 0, INSN_DIS_ALIAS }, /* jirl zero, rj, 0 */
  { 0x60000000,	0xfc00001f,	"bltz",		"r5:5,sb10:16<<2",		0,	0, 0, INSN_DIS_ALIAS }, /* blt rj, zero, offset */
  { 0x60000000,	0xfc0003e0,	"bgtz",		"r0:5,sb10:16<<2",		0,	0, 0, INSN_DIS_ALIAS }, /* blt zero, rd, offset */
  { 0x64000000,	0xfc00001f,	"bgez",		"r5:5,sb10:16<<2",		0,	0, 0, INSN_DIS_ALIAS }, /* bge rj, zero, offset */
  { 0x64000000,	0xfc0003e0,	"blez",		"r0:5,sb10:16<<2",		0,	0, 0, INSN_DIS_ALIAS }, /* bge zero, rd, offset */
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};


static struct loongarch_opcode loongarch_fix_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,			include, exclude, pinfo.  */
  { 0x0,	0x0,		"move",		"r,r",				"or %1,%2,$r0",		0,	0,	0 },
  { 0x00001000, 0xfffffc00,	"clo.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00001400, 0xfffffc00,	"clz.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00001800, 0xfffffc00,	"cto.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00001c00, 0xfffffc00,	"ctz.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00002000, 0xfffffc00,	"clo.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00002400, 0xfffffc00,	"clz.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00002800, 0xfffffc00,	"cto.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00002c00, 0xfffffc00,	"ctz.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00003000, 0xfffffc00,	"revb.2h",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00003400, 0xfffffc00,	"revb.4h",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00003800, 0xfffffc00,	"revb.2w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00003c00, 0xfffffc00,	"revb.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00004000, 0xfffffc00,	"revh.2w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00004400, 0xfffffc00,	"revh.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00004800, 0xfffffc00,	"bitrev.4b",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00004c00, 0xfffffc00,	"bitrev.8b",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00005000, 0xfffffc00,	"bitrev.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00005400, 0xfffffc00,	"bitrev.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00005800, 0xfffffc00,	"ext.w.h",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00005c00, 0xfffffc00,	"ext.w.b",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x0,	0x0,		"rdcntvl.w",	"r",				"rdtimel.w %1,$r0",	0,	0,	0 },
  { 0x0,	0x0,		"rdcntid.w",	"r",				"rdtimel.w $r0,%1",	0,	0,	0 },
  { 0x00006000, 0xfffffc00,	"rdtimel.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x0,	0x0,		"rdcntvh.w",	"r",				"rdtimeh.w %1,$r0",	0,	0,	0 },
  { 0x00006400, 0xfffffc00,	"rdtimeh.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00006800, 0xfffffc00,	"rdtime.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00006c00, 0xfffffc00,	"cpucfg",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x00010000, 0xffff801f,	"asrtle.d",	"r5:5,r10:5",			0,			0,	0,	0 },
  { 0x00018000, 0xffff801f,	"asrtgt.d",	"r5:5,r10:5",			0,			0,	0,	0 },
  { 0x00040000, 0xfffe0000,	"alsl.w",	"r0:5,r5:5,r10:5,u15:2+1",	0,			0,	0,	0 },
  { 0x00060000, 0xfffe0000,	"alsl.wu",	"r0:5,r5:5,r10:5,u15:2+1",	0,			0,	0,	0 },
  { 0x00080000, 0xfffe0000,	"bytepick.w",	"r0:5,r5:5,r10:5,u15:2",	0,			0,	0,	0 },
  { 0x000c0000, 0xfffc0000,	"bytepick.d",	"r0:5,r5:5,r10:5,u15:3",	0,			0,	0,	0 },
  { 0x00100000, 0xffff8000,	"add.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00108000, 0xffff8000,	"add.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00108000, 0xffff8000,	"add.d",	"r0:5,r5:5,r10:5,t",		0,			0,	0,	0 },
  { 0x00110000, 0xffff8000,	"sub.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00118000, 0xffff8000,	"sub.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00120000, 0xffff8000,	"slt",		"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00128000, 0xffff8000,	"sltu",		"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00130000, 0xffff8000,	"maskeqz",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00138000, 0xffff8000,	"masknez",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00140000, 0xffff8000,	"nor",		"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00148000, 0xffff8000,	"and",		"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00150000, 0xffff8000,	"or",		"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00158000, 0xffff8000,	"xor",		"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00160000, 0xffff8000,	"orn",		"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00168000, 0xffff8000,	"andn",		"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00170000, 0xffff8000,	"sll.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00178000, 0xffff8000,	"srl.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00180000, 0xffff8000,	"sra.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00188000, 0xffff8000,	"sll.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00190000, 0xffff8000,	"srl.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00198000, 0xffff8000,	"sra.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001b0000, 0xffff8000,	"rotr.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001b8000, 0xffff8000,	"rotr.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001c0000, 0xffff8000,	"mul.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001c8000, 0xffff8000,	"mulh.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001d0000, 0xffff8000,	"mulh.wu",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001d8000, 0xffff8000,	"mul.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001e0000, 0xffff8000,	"mulh.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001e8000, 0xffff8000,	"mulh.du",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001f0000, 0xffff8000,	"mulw.d.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x001f8000, 0xffff8000,	"mulw.d.wu",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00200000, 0xffff8000,	"div.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00208000, 0xffff8000,	"mod.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00210000, 0xffff8000,	"div.wu",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00218000, 0xffff8000,	"mod.wu",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00220000, 0xffff8000,	"div.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00228000, 0xffff8000,	"mod.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00230000, 0xffff8000,	"div.du",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00238000, 0xffff8000,	"mod.du",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00240000, 0xffff8000,	"crc.w.b.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00248000, 0xffff8000,	"crc.w.h.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00250000, 0xffff8000,	"crc.w.w.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00258000, 0xffff8000,	"crc.w.d.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00260000, 0xffff8000,	"crcc.w.b.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00268000, 0xffff8000,	"crcc.w.h.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00270000, 0xffff8000,	"crcc.w.w.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x00278000, 0xffff8000,	"crcc.w.d.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x002a0000, 0xffff8000,	"break",	"u0:15",			0,			0,	0,	0 },
  { 0x002a8000, 0xffff8000,	"dbcl",		"u0:15",			0,			0,	0,	0 },
  { 0x002b0000, 0xffff8000,	"syscall",	"u0:15",			0,			0,	0,	0 },
  { 0x002c0000, 0xfffe0000,	"alsl.d",	"r0:5,r5:5,r10:5,u15:2+1",	0,			0,	0,	0 },
  { 0x00408000, 0xffff8000,	"slli.w",	"r0:5,r5:5,u10:5",		0,			0,	0,	0 },
  { 0x00410000, 0xffff0000,	"slli.d",	"r0:5,r5:5,u10:6",		0,			0,	0,	0 },
  { 0x00448000, 0xffff8000,	"srli.w",	"r0:5,r5:5,u10:5",		0,			0,	0,	0 },
  { 0x00450000, 0xffff0000,	"srli.d",	"r0:5,r5:5,u10:6",		0,			0,	0,	0 },
  { 0x00488000, 0xffff8000,	"srai.w",	"r0:5,r5:5,u10:5",		0,			0,	0,	0 },
  { 0x00490000, 0xffff0000,	"srai.d",	"r0:5,r5:5,u10:6",		0,			0,	0,	0 },
  { 0x004c8000, 0xffff8000,	"rotri.w",	"r0:5,r5:5,u10:5",		0,			0,	0,	0 },
  { 0x004d0000, 0xffff0000,	"rotri.d",	"r0:5,r5:5,u10:6",		0,			0,	0,	0 },
  { 0x00600000, 0xffe08000,	"bstrins.w",	"r0:5,r5:5,u16:5,u10:5",	0,			0,	0,	0 },
  { 0x00608000, 0xffe08000,	"bstrpick.w",	"r0:5,r5:5,u16:5,u10:5",	0,			0,	0,	0 },
  { 0x00800000, 0xffc00000,	"bstrins.d",	"r0:5,r5:5,u16:6,u10:6",	0,			0,	0,	0 },
  { 0x00c00000, 0xffc00000,	"bstrpick.d",	"r0:5,r5:5,u16:6,u10:6",	0,			0,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_single_float_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,			include, exclude, pinfo.  */
  { 0x01008000, 0xffff8000,	"fadd.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01028000, 0xffff8000,	"fsub.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01048000, 0xffff8000,	"fmul.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01068000, 0xffff8000,	"fdiv.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01088000, 0xffff8000,	"fmax.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x010a8000, 0xffff8000,	"fmin.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x010c8000, 0xffff8000,	"fmaxa.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x010e8000, 0xffff8000,	"fmina.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01108000, 0xffff8000,	"fscaleb.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01128000, 0xffff8000,	"fcopysign.s",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01140400, 0xfffffc00,	"fabs.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01141400, 0xfffffc00,	"fneg.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01142400, 0xfffffc00,	"flogb.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01143400, 0xfffffc00,	"fclass.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01144400, 0xfffffc00,	"fsqrt.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01145400, 0xfffffc00,	"frecip.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01146400, 0xfffffc00,	"frsqrt.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01147400, 0xfffffc00,	"frecipe.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01148400, 0xfffffc00,	"frsqrte.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01149400, 0xfffffc00,	"fmov.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x0114a400, 0xfffffc00,	"movgr2fr.w",	"f0:5,r5:5",			0,			0,	0,	0 },
  { 0x0114ac00, 0xfffffc00,	"movgr2frh.w",	"f0:5,r5:5",			0,			0,	0,	0 },
  { 0x0114b400, 0xfffffc00,	"movfr2gr.s",	"r0:5,f5:5",			0,			0,	0,	0 },
  { 0x0114bc00, 0xfffffc00,	"movfrh2gr.s",	"r0:5,f5:5",			0,			0,	0,	0 },
  { 0x0114c000, 0xfffffc1c,	"movgr2fcsr",	"fc0:2,r5:5",			0,			0,	0,	0 },
  { 0x0114c800, 0xffffff80,	"movfcsr2gr",	"r0:5,fc5:2",			0,			0,	0,	0 },
  { 0x0114d000, 0xfffffc18,	"movfr2cf",	"c0:3,f5:5",			0,			0,	0,	0 },
  { 0x0114d400, 0xffffff00,	"movcf2fr",	"f0:5,c5:3",			0,			0,	0,	0 },
  { 0x0114d800, 0xfffffc18,	"movgr2cf",	"c0:3,r5:5",			0,			0,	0,	0 },
  { 0x0114dc00, 0xffffff00,	"movcf2gr",	"r0:5,c5:3",			0,			0,	0,	0 },
  { 0x011a0400, 0xfffffc00,	"ftintrm.w.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011a2400, 0xfffffc00,	"ftintrm.l.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011a4400, 0xfffffc00,	"ftintrp.w.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011a6400, 0xfffffc00,	"ftintrp.l.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011a8400, 0xfffffc00,	"ftintrz.w.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011aa400, 0xfffffc00,	"ftintrz.l.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011ac400, 0xfffffc00,	"ftintrne.w.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011ae400, 0xfffffc00,	"ftintrne.l.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011b0400, 0xfffffc00,	"ftint.w.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011b2400, 0xfffffc00,	"ftint.l.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011d1000, 0xfffffc00,	"ffint.s.w",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011d1800, 0xfffffc00,	"ffint.s.l",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011e4400, 0xfffffc00,	"frint.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};
static struct loongarch_opcode loongarch_double_float_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,			include, exclude, pinfo.  */
  { 0x01010000, 0xffff8000,	"fadd.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01030000, 0xffff8000,	"fsub.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01050000, 0xffff8000,	"fmul.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01070000, 0xffff8000,	"fdiv.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01090000, 0xffff8000,	"fmax.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x010b0000, 0xffff8000,	"fmin.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x010d0000, 0xffff8000,	"fmaxa.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x010f0000, 0xffff8000,	"fmina.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01110000, 0xffff8000,	"fscaleb.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01130000, 0xffff8000,	"fcopysign.d",	"f0:5,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x01140800, 0xfffffc00,	"fabs.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01141800, 0xfffffc00,	"fneg.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01142800, 0xfffffc00,	"flogb.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01143800, 0xfffffc00,	"fclass.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01144800, 0xfffffc00,	"fsqrt.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01145800, 0xfffffc00,	"frecip.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01146800, 0xfffffc00,	"frsqrt.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01147800, 0xfffffc00,	"frecipe.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01148800, 0xfffffc00,	"frsqrte.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01149800, 0xfffffc00,	"fmov.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x0114a800, 0xfffffc00,	"movgr2fr.d",	"f0:5,r5:5",			0,			0,	0,	0 },
  { 0x0114b800, 0xfffffc00,	"movfr2gr.d",	"r0:5,f5:5",			0,			0,	0,	0 },
  { 0x01191800, 0xfffffc00,	"fcvt.s.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x01192400, 0xfffffc00,	"fcvt.d.s",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011a0800, 0xfffffc00,	"ftintrm.w.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011a2800, 0xfffffc00,	"ftintrm.l.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011a4800, 0xfffffc00,	"ftintrp.w.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011a6800, 0xfffffc00,	"ftintrp.l.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011a8800, 0xfffffc00,	"ftintrz.w.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011aa800, 0xfffffc00,	"ftintrz.l.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011ac800, 0xfffffc00,	"ftintrne.w.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011ae800, 0xfffffc00,	"ftintrne.l.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011b0800, 0xfffffc00,	"ftint.w.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011b2800, 0xfffffc00,	"ftint.l.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011d2000, 0xfffffc00,	"ffint.d.w",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011d2800, 0xfffffc00,	"ffint.d.l",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0x011e4800, 0xfffffc00,	"frint.d",	"f0:5,f5:5",			0,			0,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_imm_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,			include, exclude, pinfo.  */
  { 0x02000000, 0xffc00000,	"slti",		"r0:5,r5:5,s10:12",		0,			0,	0,	0 },
  { 0x02400000, 0xffc00000,	"sltui",	"r0:5,r5:5,s10:12",		0,			0,	0,	0 },
  { 0x02800000, 0xffc00000,	"addi.w",	"r0:5,r5:5,s10:12",		0,			0,	0,	0 },
  { 0x02c00000, 0xffc00000,	"addi.d",	"r0:5,r5:5,s10:12",		0,			0,	0,	0 },
  { 0x03000000, 0xffc00000,	"lu52i.d",	"r0:5,r5:5,s10:12",		0,			0,	0,	0 },
  { 0x0,	0x0,		"nop",		"",				"andi $r0,$r0,0",	0,	0,	0 },
  { 0x03400000, 0xffc00000,	"andi",		"r0:5,r5:5,u10:12",		0,			0,	0,	0 },
  { 0x03800000, 0xffc00000,	"ori",		"r0:5,r5:5,u10:12",		0,			0,	0,	0 },
  { 0x03c00000, 0xffc00000,	"xori",		"r0:5,r5:5,u10:12",		0,			0,	0,	0 },
  { 0x10000000, 0xfc000000,	"addu16i.d",	"r0:5,r5:5,s10:16",		0,			0,	0,	0 },
  { 0x14000000, 0xfe000000,	"lu12i.w",	"r0:5,s5:20",			0,			0,	0,	0 },
  { 0x16000000, 0xfe000000,	"lu32i.d",	"r0:5,s5:20",			0,			0,	0,	0 },
  { 0x18000000, 0xfe000000,	"pcaddi",	"r0:5,s5:20",			0,			0,	0,	0 },
  { 0x1a000000, 0xfe000000,	"pcalau12i",	"r0:5,s5:20",			0,			0,	0,	0 },
  { 0x1c000000, 0xfe000000,	"pcaddu12i",	"r0:5,s5:20",			0,			0,	0,	0 },
  { 0x1e000000, 0xfe000000,	"pcaddu18i",	"r0:5,s5:20",			0,			0,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_privilege_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,			include, exclude, pinfo.  */
  { 0x04000000, 0xff0003e0,	"csrrd",	"r0:5,u10:14",			0,			0,	0,	0 },
  { 0x04000020, 0xff0003e0,	"csrwr",	"r0:5,u10:14",			0,			0,	0,	0 },
  { 0x04000000, 0xff000000,	"csrxchg",	"r0:5,r5:5,u10:14",		0,			0,	0,	0 },
  { 0x06000000, 0xffc00000,	"cacop",	"u0:5,r5:5,s10:12",		0,			0,	0,	0 },
  { 0x06400000, 0xfffc0000,	"lddir",	"r0:5,r5:5,u10:8",		0,			0,	0,	0 },
  { 0x06440000, 0xfffc001f,	"ldpte",	"r5:5,u10:8",			0,			0,	0,	0 },
  { 0x06480000, 0xfffffc00,	"iocsrrd.b",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x06480400, 0xfffffc00,	"iocsrrd.h",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x06480800, 0xfffffc00,	"iocsrrd.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x06480c00, 0xfffffc00,	"iocsrrd.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x06481000, 0xfffffc00,	"iocsrwr.b",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x06481400, 0xfffffc00,	"iocsrwr.h",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x06481800, 0xfffffc00,	"iocsrwr.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x06481c00, 0xfffffc00,	"iocsrwr.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x06482000, 0xffffffff,	"tlbclr",	"",				0,			0,	0,	0 },
  { 0x06482400, 0xffffffff,	"tlbflush",	"",				0,			0,	0,	0 },
  { 0x06482800, 0xffffffff,	"tlbsrch",	"",				0,			0,	0,	0 },
  { 0x06482c00, 0xffffffff,	"tlbrd",	"",				0,			0,	0,	0 },
  { 0x06483000, 0xffffffff,	"tlbwr",	"",				0,			0,	0,	0 },
  { 0x06483400, 0xffffffff,	"tlbfill",	"",				0,			0,	0,	0 },
  { 0x06483800, 0xffffffff,	"ertn",		"",				0,			0,	0,	0 },
  { 0x06488000, 0xffff8000,	"idle",		"u0:15",			0,			0,	0,	0 },
  { 0x06498000, 0xffff8000,	"invtlb",	"u0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_4opt_single_float_opcodes[] =
{
  /* match,	mask,		name,			format,				macro,	include, exclude, pinfo.  */
  { 0x08100000, 0xfff00000,	"fmadd.s",		"f0:5,f5:5,f10:5,f15:5",	0,	0,	0,	0 },
  { 0x08500000, 0xfff00000,	"fmsub.s",		"f0:5,f5:5,f10:5,f15:5",	0,	0,	0,	0 },
  { 0x08900000, 0xfff00000,	"fnmadd.s",		"f0:5,f5:5,f10:5,f15:5",	0,	0,	0,	0 },
  { 0x08d00000, 0xfff00000,	"fnmsub.s",		"f0:5,f5:5,f10:5,f15:5",	0,	0,	0,	0 },
  { 0x09100000, 0xfff00000,	"vfmadd.s",		"v0:5,v5:5,v10:5,v15:5",	0,	0,	0,	0 },
  { 0x09500000, 0xfff00000,	"vfmsub.s",		"v0:5,v5:5,v10:5,v15:5",	0,	0,	0,	0 },
  { 0x09900000, 0xfff00000,	"vfnmadd.s",		"v0:5,v5:5,v10:5,v15:5",	0,	0,	0,	0 },
  { 0x09d00000, 0xfff00000,	"vfnmsub.s",		"v0:5,v5:5,v10:5,v15:5",	0,	0,	0,	0 },
  { 0x0a100000, 0xfff00000,	"xvfmadd.s",		"x0:5,x5:5,x10:5,x15:5",	0,	0,	0,	0 },
  { 0x0a500000, 0xfff00000,	"xvfmsub.s",		"x0:5,x5:5,x10:5,x15:5",	0,	0,	0,	0 },
  { 0x0a900000, 0xfff00000,	"xvfnmadd.s",		"x0:5,x5:5,x10:5,x15:5",	0,	0,	0,	0 },
  { 0x0ad00000, 0xfff00000,	"xvfnmsub.s",		"x0:5,x5:5,x10:5,x15:5",	0,	0,	0,	0 },
  { 0x0c100000, 0xffff8018,	"fcmp.caf.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c108000, 0xffff8018,	"fcmp.saf.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c110000, 0xffff8018,	"fcmp.clt.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c118000, 0xffff8018,	"fcmp.slt.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c118000, 0xffff8018,	"fcmp.sgt.s",		"c0:3,f10:5,f5:5",		0,	0,	0,	0 },
  { 0x0c120000, 0xffff8018,	"fcmp.ceq.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c128000, 0xffff8018,	"fcmp.seq.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c130000, 0xffff8018,	"fcmp.cle.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c138000, 0xffff8018,	"fcmp.sle.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c138000, 0xffff8018,	"fcmp.sge.s",		"c0:3,f10:5,f5:5",		0,	0,	0,	0 },
  { 0x0c140000, 0xffff8018,	"fcmp.cun.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c148000, 0xffff8018,	"fcmp.sun.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c150000, 0xffff8018,	"fcmp.cult.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c150000, 0xffff8018,	"fcmp.cugt.s",		"c0:3,f10:5,f5:5",		0,	0,	0,	0 },
  { 0x0c158000, 0xffff8018,	"fcmp.sult.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c160000, 0xffff8018,	"fcmp.cueq.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c168000, 0xffff8018,	"fcmp.sueq.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c170000, 0xffff8018,	"fcmp.cule.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c170000, 0xffff8018,	"fcmp.cuge.s",		"c0:3,f10:5,f5:5",		0,	0,	0,	0 },
  { 0x0c178000, 0xffff8018,	"fcmp.sule.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c180000, 0xffff8018,	"fcmp.cne.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c188000, 0xffff8018,	"fcmp.sne.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c1a0000, 0xffff8018,	"fcmp.cor.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c1a8000, 0xffff8018,	"fcmp.sor.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c1c0000, 0xffff8018,	"fcmp.cune.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c1c8000, 0xffff8018,	"fcmp.sune.s",		"c0:3,f5:5,f10:5",		0,	0,	0,	0 },
  { 0x0c500000, 0xffff8000,	"vfcmp.caf.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c508000, 0xffff8000,	"vfcmp.saf.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c510000, 0xffff8000,	"vfcmp.clt.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c518000, 0xffff8000,	"vfcmp.slt.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c520000, 0xffff8000,	"vfcmp.ceq.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c528000, 0xffff8000,	"vfcmp.seq.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c530000, 0xffff8000,	"vfcmp.cle.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c538000, 0xffff8000,	"vfcmp.sle.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c540000, 0xffff8000,	"vfcmp.cun.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c548000, 0xffff8000,	"vfcmp.sun.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c550000, 0xffff8000,	"vfcmp.cult.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c558000, 0xffff8000,	"vfcmp.sult.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c560000, 0xffff8000,	"vfcmp.cueq.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c568000, 0xffff8000,	"vfcmp.sueq.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c570000, 0xffff8000,	"vfcmp.cule.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c578000, 0xffff8000,	"vfcmp.sule.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c580000, 0xffff8000,	"vfcmp.cne.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c588000, 0xffff8000,	"vfcmp.sne.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c5a0000, 0xffff8000,	"vfcmp.cor.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c5a8000, 0xffff8000,	"vfcmp.sor.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c5c0000, 0xffff8000,	"vfcmp.cune.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c5c8000, 0xffff8000,	"vfcmp.sune.s",		"v0:5,v5:5,v10:5",		0,	0,	0,	0 },
  { 0x0c900000, 0xffff8000,	"xvfcmp.caf.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c908000, 0xffff8000,	"xvfcmp.saf.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c910000, 0xffff8000,	"xvfcmp.clt.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c918000, 0xffff8000,	"xvfcmp.slt.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c920000, 0xffff8000,	"xvfcmp.ceq.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c928000, 0xffff8000,	"xvfcmp.seq.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c930000, 0xffff8000,	"xvfcmp.cle.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c938000, 0xffff8000,	"xvfcmp.sle.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c940000, 0xffff8000,	"xvfcmp.cun.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c948000, 0xffff8000,	"xvfcmp.sun.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c950000, 0xffff8000,	"xvfcmp.cult.s",	"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c958000, 0xffff8000,	"xvfcmp.sult.s",	"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c960000, 0xffff8000,	"xvfcmp.cueq.s",	"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c968000, 0xffff8000,	"xvfcmp.sueq.s",	"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c970000, 0xffff8000,	"xvfcmp.cule.s",	"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c978000, 0xffff8000,	"xvfcmp.sule.s",	"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c980000, 0xffff8000,	"xvfcmp.cne.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c988000, 0xffff8000,	"xvfcmp.sne.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c9a0000, 0xffff8000,	"xvfcmp.cor.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c9a8000, 0xffff8000,	"xvfcmp.sor.s",		"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c9c0000, 0xffff8000,	"xvfcmp.cune.s",	"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0c9c8000, 0xffff8000,	"xvfcmp.sune.s",	"x0:5,x5:5,x10:5",		0,	0,	0,	0 },
  { 0x0d000000, 0xfffc0000,	"fsel",			"f0:5,f5:5,f10:5,c15:3",	0,	0,	0,	0 },
  { 0x0d100000, 0xfff00000,	"vbitsel.v",		"v0:5,v5:5,v10:5,v15:5",	0,	0,	0,	0 },
  { 0x0d200000, 0xfff00000,	"xvbitsel.v",		"x0:5,x5:5,x10:5,x15:5",	0,	0,	0,	0 },
  { 0x0d500000, 0xfff00000,	"vshuf.b",		"v0:5,v5:5,v10:5,v15:5",	0,	0,	0,	0 },
  { 0x0d600000, 0xfff00000,	"xvshuf.b",		"x0:5,x5:5,x10:5,x15:5",	0,	0,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_4opt_double_float_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,			include, exclude, pinfo.  */
  { 0x08200000, 0xfff00000,	"fmadd.d",	"f0:5,f5:5,f10:5,f15:5",	0,			0,	0,	0 },
  { 0x08600000, 0xfff00000,	"fmsub.d",	"f0:5,f5:5,f10:5,f15:5",	0,			0,	0,	0 },
  { 0x08a00000, 0xfff00000,	"fnmadd.d",	"f0:5,f5:5,f10:5,f15:5",	0,			0,	0,	0 },
  { 0x08e00000, 0xfff00000,	"fnmsub.d",	"f0:5,f5:5,f10:5,f15:5",	0,			0,	0,	0 },
  { 0x09200000, 0xfff00000,	"vfmadd.d",	"v0:5,v5:5,v10:5,v15:5",	0,			0,	0,	0 },
  { 0x09600000, 0xfff00000,	"vfmsub.d",	"v0:5,v5:5,v10:5,v15:5",	0, 			0,	0,	0 },
  { 0x09a00000, 0xfff00000,	"vfnmadd.d",	"v0:5,v5:5,v10:5,v15:5",	0, 			0,	0,	0 },
  { 0x09e00000, 0xfff00000,	"vfnmsub.d",	"v0:5,v5:5,v10:5,v15:5",	0, 			0,	0,	0 },
  { 0x0a200000, 0xfff00000,	"xvfmadd.d",	"x0:5,x5:5,x10:5,x15:5",	0, 			0,	0,	0 },
  { 0x0a600000, 0xfff00000,	"xvfmsub.d",	"x0:5,x5:5,x10:5,x15:5",	0, 			0,	0,	0 },
  { 0x0aa00000, 0xfff00000,	"xvfnmadd.d",	"x0:5,x5:5,x10:5,x15:5",	0, 			0,	0,	0 },
  { 0x0ae00000, 0xfff00000,	"xvfnmsub.d",	"x0:5,x5:5,x10:5,x15:5",	0, 			0,	0,	0 },
  { 0x0c200000, 0xffff8018,	"fcmp.caf.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c208000, 0xffff8018,	"fcmp.saf.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c210000, 0xffff8018,	"fcmp.clt.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c218000, 0xffff8018,	"fcmp.slt.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c218000, 0xffff8018,	"fcmp.sgt.d",	"c0:3,f10:5,f5:5",		0,			0,	0,	0 },
  { 0x0c220000, 0xffff8018,	"fcmp.ceq.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c228000, 0xffff8018,	"fcmp.seq.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c230000, 0xffff8018,	"fcmp.cle.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c238000, 0xffff8018,	"fcmp.sle.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c238000, 0xffff8018,	"fcmp.sge.d",	"c0:3,f10:5,f5:5",		0,			0,	0,	0 },
  { 0x0c240000, 0xffff8018,	"fcmp.cun.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c248000, 0xffff8018,	"fcmp.sun.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c250000, 0xffff8018,	"fcmp.cult.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c250000, 0xffff8018,	"fcmp.cugt.d",	"c0:3,f10:5,f5:5",		0,			0,	0,	0 },
  { 0x0c258000, 0xffff8018,	"fcmp.sult.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c260000, 0xffff8018,	"fcmp.cueq.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c268000, 0xffff8018,	"fcmp.sueq.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c270000, 0xffff8018,	"fcmp.cule.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c270000, 0xffff8018,	"fcmp.cuge.d",	"c0:3,f10:5,f5:5",		0,			0,	0,	0 },
  { 0x0c278000, 0xffff8018,	"fcmp.sule.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c280000, 0xffff8018,	"fcmp.cne.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c288000, 0xffff8018,	"fcmp.sne.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c2a0000, 0xffff8018,	"fcmp.cor.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c2a8000, 0xffff8018,	"fcmp.sor.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c2c0000, 0xffff8018,	"fcmp.cune.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c2c8000, 0xffff8018,	"fcmp.sune.d",	"c0:3,f5:5,f10:5",		0,			0,	0,	0 },
  { 0x0c600000, 0xffff8000, "vfcmp.caf.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c608000, 0xffff8000, "vfcmp.saf.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c610000, 0xffff8000, "vfcmp.clt.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c618000, 0xffff8000, "vfcmp.slt.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c620000, 0xffff8000, "vfcmp.ceq.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c628000, 0xffff8000, "vfcmp.seq.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c630000, 0xffff8000, "vfcmp.cle.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c638000, 0xffff8000, "vfcmp.sle.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c640000, 0xffff8000, "vfcmp.cun.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c648000, 0xffff8000, "vfcmp.sun.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c650000, 0xffff8000, "vfcmp.cult.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c658000, 0xffff8000, "vfcmp.sult.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c660000, 0xffff8000, "vfcmp.cueq.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c668000, 0xffff8000, "vfcmp.sueq.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c670000, 0xffff8000, "vfcmp.cule.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c678000, 0xffff8000, "vfcmp.sule.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c680000, 0xffff8000, "vfcmp.cne.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c688000, 0xffff8000, "vfcmp.sne.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c6a0000, 0xffff8000, "vfcmp.cor.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c6a8000, 0xffff8000, "vfcmp.sor.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c6c0000, 0xffff8000, "vfcmp.cune.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0c6c8000, 0xffff8000, "vfcmp.sune.d", "v0:5,v5:5,v10:5",			0,			0,	0,	0 },
  { 0x0ca00000, 0xffff8000, "xvfcmp.caf.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca08000, 0xffff8000, "xvfcmp.saf.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca10000, 0xffff8000, "xvfcmp.clt.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca18000, 0xffff8000, "xvfcmp.slt.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca20000, 0xffff8000, "xvfcmp.ceq.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca28000, 0xffff8000, "xvfcmp.seq.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca30000, 0xffff8000, "xvfcmp.cle.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca38000, 0xffff8000, "xvfcmp.sle.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca40000, 0xffff8000, "xvfcmp.cun.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca48000, 0xffff8000, "xvfcmp.sun.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca50000, 0xffff8000, "xvfcmp.cult.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca58000, 0xffff8000, "xvfcmp.sult.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca60000, 0xffff8000, "xvfcmp.cueq.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca68000, 0xffff8000, "xvfcmp.sueq.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca70000, 0xffff8000, "xvfcmp.cule.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca78000, 0xffff8000, "xvfcmp.sule.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca80000, 0xffff8000, "xvfcmp.cne.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0ca88000, 0xffff8000, "xvfcmp.sne.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0caa0000, 0xffff8000, "xvfcmp.cor.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0caa8000, 0xffff8000, "xvfcmp.sor.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0cac0000, 0xffff8000, "xvfcmp.cune.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0x0cac8000, 0xffff8000, "xvfcmp.sune.d", "x0:5,x5:5,x10:5",			0,			0,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_load_store_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,			include, exclude, pinfo.  */
  { 0x20000000, 0xff000000,	"ll.w",		"r0:5,r5:5,so10:14<<2",		0,			0,	0,	0 },
  { 0x21000000, 0xff000000,	"sc.w",		"r0:5,r5:5,so10:14<<2",		0,			0,	0,	0 },
  { 0x22000000, 0xff000000,	"ll.d",		"r0:5,r5:5,so10:14<<2",		0,			0,	0,	0 },
  { 0x23000000, 0xff000000,	"sc.d",		"r0:5,r5:5,so10:14<<2",		0,			0,	0,	0 },
  { 0x24000000, 0xff000000,	"ldptr.w",	"r0:5,r5:5,so10:14<<2",		0,			0,	0,	0 },
  { 0x25000000, 0xff000000,	"stptr.w",	"r0:5,r5:5,so10:14<<2",		0,			0,	0,	0 },
  { 0x26000000, 0xff000000,	"ldptr.d",	"r0:5,r5:5,so10:14<<2",		0,			0,	0,	0 },
  { 0x27000000, 0xff000000,	"stptr.d",	"r0:5,r5:5,so10:14<<2",		0,			0,	0,	0 },
  { 0x28000000, 0xffc00000,	"ld.b",		"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x28400000, 0xffc00000,	"ld.h",		"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x28800000, 0xffc00000,	"ld.w",		"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x28c00000, 0xffc00000,	"ld.d",		"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x29000000, 0xffc00000,	"st.b",		"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x29400000, 0xffc00000,	"st.h",		"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x29800000, 0xffc00000,	"st.w",		"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x29c00000, 0xffc00000,	"st.d",		"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x2a000000, 0xffc00000,	"ld.bu",	"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x2a400000, 0xffc00000,	"ld.hu",	"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x2a800000, 0xffc00000,	"ld.wu",	"r0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x2ac00000, 0xffc00000,	"preld",	"u0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x38000000, 0xffff8000,	"ldx.b",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38040000, 0xffff8000,	"ldx.h",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38080000, 0xffff8000,	"ldx.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x380c0000, 0xffff8000,	"ldx.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38100000, 0xffff8000,	"stx.b",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38140000, 0xffff8000,	"stx.h",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38180000, 0xffff8000,	"stx.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x381c0000, 0xffff8000,	"stx.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38200000, 0xffff8000,	"ldx.bu",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38240000, 0xffff8000,	"ldx.hu",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38280000, 0xffff8000,	"ldx.wu",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x382c0000, 0xffff8000,	"preldx",	"u0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"sc.q",		"r,r,r,u0:0",			"sc.q %1,%2,%3",	0,	0,	0 },
  { 0x38570000, 0xffff8000,	"sc.q",		"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"llacq.w",	"r,r,u0:0",			"llacq.w %1,%2",	0,	0,	0 },
  { 0x38578000, 0xfffffc00,	"llacq.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x0,	0x0,		"screl.w",	"r,r,u0:0",			"screl.w %1,%2",	0,	0,	0 },
  { 0x38578400, 0xfffffc00,	"screl.w",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x0,	0x0,		"llacq.d",	"r,r,u0:0",			"llacq.d %1,%2",	0,	0,	0 },
  { 0x38578800, 0xfffffc00,	"llacq.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x0,	0x0,		"screl.d",	"r,r,u0:0",			"screl.d %1,%2",	0,	0,	0 },
  { 0x38578c00, 0xfffffc00,	"screl.d",	"r0:5,r5:5",			0,			0,	0,	0 },
  { 0x0,	0x0,		"amcas.b",	"r,r,r,u0:0",			"amcas.b %1,%2,%3",	0,	0,	0 },
  { 0x38580000, 0xffff8000,	"amcas.b",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amcas.h",	"r,r,r,u0:0",			"amcas.h %1,%2,%3",	0,	0,	0 },
  { 0x38588000, 0xffff8000,	"amcas.h",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amcas.w",	"r,r,r,u0:0",			"amcas.w %1,%2,%3",	0,	0,	0 },
  { 0x38590000, 0xffff8000,	"amcas.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amcas.d",	"r,r,r,u0:0",			"amcas.d %1,%2,%3",	0,	0,	0 },
  { 0x38598000, 0xffff8000,	"amcas.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amcas_db.b",	"r,r,r,u0:0",			"amcas_db.b %1,%2,%3",	0,	0,	0 },
  { 0x385a0000, 0xffff8000,	"amcas_db.b",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amcas_db.h",	"r,r,r,u0:0",			"amcas_db.h %1,%2,%3",	0,	0,	0 },
  { 0x385a8000, 0xffff8000,	"amcas_db.h",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amcas_db.w",	"r,r,r,u0:0",			"amcas_db.w %1,%2,%3",	0,	0,	0 },
  { 0x385b0000, 0xffff8000,	"amcas_db.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amcas_db.d",	"r,r,r,u0:0",			"amcas_db.d %1,%2,%3",	0,	0,	0 },
  { 0x385b8000, 0xffff8000,	"amcas_db.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amswap.b",	"r,r,r,u0:0",			"amswap.b %1,%2,%3",	0,	0,	0 },
  { 0x385c0000, 0xffff8000,	"amswap.b",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amswap.h",	"r,r,r,u0:0",			"amswap.h %1,%2,%3",	0,	0,	0 },
  { 0x385c8000, 0xffff8000,	"amswap.h",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amadd.b",	"r,r,r,u0:0",			"amadd.b %1,%2,%3",	0,	0,	0 },
  { 0x385d0000, 0xffff8000,	"amadd.b",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amadd.h",	"r,r,r,u0:0",			"amadd.h %1,%2,%3",	0,	0,	0 },
  { 0x385d8000, 0xffff8000,	"amadd.h",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amswap_db.b",	"r,r,r,u0:0",			"amswap_db.b %1,%2,%3",	0,	0,	0 },
  { 0x385e0000, 0xffff8000,	"amswap_db.b",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amswap_db.h",	"r,r,r,u0:0",			"amswap_db.h %1,%2,%3",	0,	0,	0 },
  { 0x385e8000, 0xffff8000,	"amswap_db.h",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amadd_db.b",	"r,r,r,u0:0",			"amadd_db.b %1,%2,%3",	0,	0,	0 },
  { 0x385f0000, 0xffff8000,	"amadd_db.b",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amadd_db.h",	"r,r,r,u0:0",			"amadd_db.h %1,%2,%3",	0,	0,	0 },
  { 0x385f8000, 0xffff8000,	"amadd_db.h",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amswap.w",	"r,r,r,u0:0",			"amswap.w %1,%2,%3",	0,	0,	0 },
  { 0x38600000, 0xffff8000,	"amswap.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amswap.d",	"r,r,r,u0:0",			"amswap.d %1,%2,%3",	0,	0,	0 },
  { 0x38608000, 0xffff8000,	"amswap.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amadd.w",	"r,r,r,u0:0",			"amadd.w %1,%2,%3",	0,	0,	0 },
  { 0x38610000, 0xffff8000,	"amadd.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amadd.d",	"r,r,r,u0:0",			"amadd.d %1,%2,%3",	0,	0,	0 },
  { 0x38618000, 0xffff8000,	"amadd.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amand.w",	"r,r,r,u0:0",			"amand.w %1,%2,%3",	0,	0,	0 },
  { 0x38620000, 0xffff8000,	"amand.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amand.d",	"r,r,r,u0:0",			"amand.d %1,%2,%3",	0,	0,	0 },
  { 0x38628000, 0xffff8000,	"amand.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amor.w",	"r,r,r,u0:0",			"amor.w %1,%2,%3",	0,	0,	0 },
  { 0x38630000, 0xffff8000,	"amor.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amor.d",	"r,r,r,u0:0",			"amor.d %1,%2,%3",	0,	0,	0 },
  { 0x38638000, 0xffff8000,	"amor.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amxor.w",	"r,r,r,u0:0",			"amxor.w %1,%2,%3",	0,	0,	0 },
  { 0x38640000, 0xffff8000,	"amxor.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amxor.d",	"r,r,r,u0:0",			"amxor.d %1,%2,%3",	0,	0,	0 },
  { 0x38648000, 0xffff8000,	"amxor.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammax.w",	"r,r,r,u0:0",			"ammax.w %1,%2,%3",	0,	0,	0 },
  { 0x38650000, 0xffff8000,	"ammax.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammax.d",	"r,r,r,u0:0",			"ammax.d %1,%2,%3",	0,	0,	0 },
  { 0x38658000, 0xffff8000,	"ammax.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammin.w",	"r,r,r,u0:0",			"ammin.w %1,%2,%3",	0,	0,	0 },
  { 0x38660000, 0xffff8000,	"ammin.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammin.d",	"r,r,r,u0:0",			"ammin.d %1,%2,%3",	0,	0,	0 },
  { 0x38668000, 0xffff8000,	"ammin.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammax.wu",	"r,r,r,u0:0",			"ammax.wu %1,%2,%3",	0,	0,	0 },
  { 0x38670000, 0xffff8000,	"ammax.wu",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammax.du",	"r,r,r,u0:0",			"ammax.du %1,%2,%3",	0,	0,	0 },
  { 0x38678000, 0xffff8000,	"ammax.du",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammin.wu",	"r,r,r,u0:0",			"ammin.wu %1,%2,%3",	0,	0,	0 },
  { 0x38680000, 0xffff8000,	"ammin.wu",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammin.du",	"r,r,r,u0:0",			"ammin.du %1,%2,%3",	0,	0,	0 },
  { 0x38688000, 0xffff8000,	"ammin.du",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amswap_db.w",	"r,r,r,u0:0",			"amswap_db.w %1,%2,%3",	0,	0,	0 },
  { 0x38690000, 0xffff8000,	"amswap_db.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amswap_db.d",	"r,r,r,u0:0",			"amswap_db.d %1,%2,%3",	0,	0,	0 },
  { 0x38698000, 0xffff8000,	"amswap_db.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amadd_db.w",	"r,r,r,u0:0",			"amadd_db.w %1,%2,%3",	0,	0,	0 },
  { 0x386a0000, 0xffff8000,	"amadd_db.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amadd_db.d",	"r,r,r,u0:0",			"amadd_db.d %1,%2,%3",	0,	0,	0 },
  { 0x386a8000, 0xffff8000,	"amadd_db.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amand_db.w",	"r,r,r,u0:0",			"amand_db.w %1,%2,%3",	0,	0,	0 },
  { 0x386b0000, 0xffff8000,	"amand_db.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amand_db.d",	"r,r,r,u0:0",			"amand_db.d %1,%2,%3",	0,	0,	0 },
  { 0x386b8000, 0xffff8000,	"amand_db.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amor_db.w",	"r,r,r,u0:0",			"amor_db.w %1,%2,%3",	0,	0,	0 },
  { 0x386c0000, 0xffff8000,	"amor_db.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amor_db.d",	"r,r,r,u0:0",			"amor_db.d %1,%2,%3",	0,	0,	0 },
  { 0x386c8000, 0xffff8000,	"amor_db.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amxor_db.w",	"r,r,r,u0:0",			"amxor_db.w %1,%2,%3",	0,	0,	0 },
  { 0x386d0000, 0xffff8000,	"amxor_db.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"amxor_db.d",	"r,r,r,u0:0",			"amxor_db.d %1,%2,%3",	0,	0,	0 },
  { 0x386d8000, 0xffff8000,	"amxor_db.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammax_db.w",	"r,r,r,u0:0",			"ammax_db.w %1,%2,%3",	0,	0,	0 },
  { 0x386e0000, 0xffff8000,	"ammax_db.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammax_db.d",	"r,r,r,u0:0",			"ammax_db.d %1,%2,%3",	0,	0,	0 },
  { 0x386e8000, 0xffff8000,	"ammax_db.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammin_db.w",	"r,r,r,u0:0",			"ammin_db.w %1,%2,%3",	0,	0,	0 },
  { 0x386f0000, 0xffff8000,	"ammin_db.w",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammin_db.d",	"r,r,r,u0:0",			"ammin_db.d %1,%2,%3",	0,	0,	0 },
  { 0x386f8000, 0xffff8000,	"ammin_db.d",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammax_db.wu",	"r,r,r,u0:0",			"ammax_db.wu %1,%2,%3",	0,	0,	0 },
  { 0x38700000, 0xffff8000,	"ammax_db.wu",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammax_db.du",	"r,r,r,u0:0",			"ammax_db.du %1,%2,%3",	0,	0,	0 },
  { 0x38708000, 0xffff8000,	"ammax_db.du",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammin_db.wu",	"r,r,r,u0:0",			"ammin_db.wu %1,%2,%3",	0,	0,	0 },
  { 0x38710000, 0xffff8000,	"ammin_db.wu",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x0,	0x0,		"ammin_db.du",	"r,r,r,u0:0",			"ammin_db.du %1,%2,%3",	0,	0,	0 },
  { 0x38718000, 0xffff8000,	"ammin_db.du",	"r0:5,r10:5,r5:5",		0,			0,	0,	0 },
  { 0x38720000, 0xffff8000,	"dbar",		"u0:15",			0,			0,	0,	0 },
  { 0x38728000, 0xffff8000,	"ibar",		"u0:15",			0,			0,	0,	0 },
  { 0x38780000, 0xffff8000,	"ldgt.b",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38788000, 0xffff8000,	"ldgt.h",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38790000, 0xffff8000,	"ldgt.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38798000, 0xffff8000,	"ldgt.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387a0000, 0xffff8000,	"ldle.b",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387a8000, 0xffff8000,	"ldle.h",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387b0000, 0xffff8000,	"ldle.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387b8000, 0xffff8000,	"ldle.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387c0000, 0xffff8000,	"stgt.b",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387c8000, 0xffff8000,	"stgt.h",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387d0000, 0xffff8000,	"stgt.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387d8000, 0xffff8000,	"stgt.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387e0000, 0xffff8000,	"stle.b",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387e8000, 0xffff8000,	"stle.h",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387f0000, 0xffff8000,	"stle.w",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x387f8000, 0xffff8000,	"stle.d",	"r0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x2c000000, 0xffc00000,	"vld",		"v0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x2c400000, 0xffc00000,	"vst",		"v0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x2c800000, 0xffc00000,	"xvld",		"x0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x2cc00000, 0xffc00000,	"xvst",		"x0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x38400000, 0xffff8000,	"vldx",		"v0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38440000, 0xffff8000,	"vstx",		"v0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x38480000, 0xffff8000,	"xvldx",	"x0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x384c0000, 0xffff8000,	"xvstx",	"x0:5,r5:5,r10:5",		0,			0,	0,	0 },
  { 0x30100000, 0xfff80000,	"vldrepl.d",	"v0:5,r5:5,so10:9<<3",		0,			0,	0,	0 },
  { 0x30200000, 0xfff00000,	"vldrepl.w",	"v0:5,r5:5,so10:10<<2",		0,			0,	0,	0 },
  { 0x30400000, 0xffe00000,	"vldrepl.h",	"v0:5,r5:5,so10:11<<1",		0,			0,	0,	0 },
  { 0x30800000, 0xffc00000,	"vldrepl.b",	"v0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x31100000, 0xfff80000,	"vstelm.d",	"v0:5,r5:5,so10:8<<3,u18:1",	0,			0,	0,	0 },
  { 0x31200000, 0xfff00000,	"vstelm.w",	"v0:5,r5:5,so10:8<<2,u18:2",	0,			0,	0,	0 },
  { 0x31400000, 0xffe00000,	"vstelm.h",	"v0:5,r5:5,so10:8<<1,u18:3",	0,			0,	0,	0 },
  { 0x31800000, 0xffc00000,	"vstelm.b",	"v0:5,r5:5,so10:8,u18:4",	0,			0,	0,	0 },
  { 0x32100000, 0xfff80000,	"xvldrepl.d",	"x0:5,r5:5,so10:9<<3",		0,			0,	0,	0 },
  { 0x32200000, 0xfff00000,	"xvldrepl.w",	"x0:5,r5:5,so10:10<<2",		0,			0,	0,	0 },
  { 0x32400000, 0xffe00000,	"xvldrepl.h",	"x0:5,r5:5,so10:11<<1",		0,			0,	0,	0 },
  { 0x32800000, 0xffc00000,	"xvldrepl.b",	"x0:5,r5:5,so10:12",		0,			0,	0,	0 },
  { 0x33100000, 0xfff00000,	"xvstelm.d",	"x0:5,r5:5,so10:8<<3,u18:2",	0,			0,	0,	0 },
  { 0x33200000, 0xffe00000,	"xvstelm.w",	"x0:5,r5:5,so10:8<<2,u18:3",	0,			0,	0,	0 },
  { 0x33400000, 0xffc00000,	"xvstelm.h",	"x0:5,r5:5,so10:8<<1,u18:4",	0,			0,	0,	0 },
  { 0x33800000, 0xff800000,	"xvstelm.b",	"x0:5,r5:5,so10:8,u18:5",	0,			0,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_single_float_load_store_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,	include,		exclude, pinfo.  */
  { 0x2b000000, 0xffc00000,	"fld.s",	"f0:5,r5:5,so10:12",		0,	0,			0,	0 },
  { 0x2b400000, 0xffc00000,	"fst.s",	"f0:5,r5:5,so10:12",		0,	0,			0,	0 },
  { 0x38300000, 0xffff8000,	"fldx.s",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x38380000, 0xffff8000,	"fstx.s",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x38740000, 0xffff8000,	"fldgt.s",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x38750000, 0xffff8000,	"fldle.s",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x38760000, 0xffff8000,	"fstgt.s",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x38770000, 0xffff8000,	"fstle.s",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_double_float_load_store_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,	include,		exclude, pinfo.  */
  { 0x2b800000, 0xffc00000,	"fld.d",	"f0:5,r5:5,so10:12",		0,	0,			0,	0 },
  { 0x2bc00000, 0xffc00000,	"fst.d",	"f0:5,r5:5,so10:12",		0,	0,			0,	0 },
  { 0x38340000, 0xffff8000,	"fldx.d",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x383c0000, 0xffff8000,	"fstx.d",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x38748000, 0xffff8000,	"fldgt.d",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x38758000, 0xffff8000,	"fldle.d",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x38768000, 0xffff8000,	"fstgt.d",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0x38778000, 0xffff8000,	"fstle.d",	"f0:5,r5:5,r10:5",		0,	&LARCH_opts.ase_lp64,	0,	0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_float_jmp_opcodes[] =
{
  { 0x0,	0x0,		"bceqz",	"c,la",				"bceqz %1,%%b21(%2)",		0, 0, 0 },
  { 0x48000000, 0xfc000300,	"bceqz",	"c5:3,sb0:5|10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"bcnez",	"c,la",				"bcnez %1,%%b21(%2)",		0, 0, 0 },
  { 0x48000100, 0xfc000300,	"bcnez",	"c5:3,sb0:5|10:16<<2",		0,				0, 0, 0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_jmp_opcodes[] =
{
  /* match,	mask,		name,		format,				macro,			include, exclude, pinfo.  */
  { 0x0,	0x0,		"beqz",		"r,la",				"beqz %1,%%b21(%2)",		0, 0, 0 },
  { 0x40000000, 0xfc000000,	"beqz",		"r5:5,sb0:5|10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"bnez",		"r,la",				"bnez %1,%%b21(%2)",		0, 0, 0 },
  { 0x44000000, 0xfc000000,	"bnez",		"r5:5,sb0:5|10:16<<2",		0,				0, 0, 0 },
  { 0x4c000000, 0xfc000000,	"jirl",		"r0:5,r5:5,so10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"b",		"la",				"b %%b26(%1)",			0, 0, 0 },
  { 0x50000000, 0xfc000000,	"b",		"sb0:10|10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"bl",		"la",				"bl %%b26(%1)",			0, 0, 0 },
  { 0x54000000, 0xfc000000,	"bl",		"sb0:10|10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"beq",		"r,r,la",			"beq %1,%2,%%b16(%3)",		0, 0, 0 },
  { 0x58000000, 0xfc000000,	"beq",		"r5:5,r0:5,sb10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"bne",		"r,r,la",			"bne %1,%2,%%b16(%3)",		0, 0, 0 },
  { 0x5c000000, 0xfc000000,	"bne",		"r5:5,r0:5,sb10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"blt",		"r,r,la",			"blt %1,%2,%%b16(%3)",		0, 0, 0 },
  { 0x60000000, 0xfc000000,	"blt",		"r5:5,r0:5,sb10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"bge",		"r,r,la",			"bge %1,%2,%%b16(%3)",		0, 0, 0 },
  { 0x64000000, 0xfc000000,	"bge",		"r5:5,r0:5,sb10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"bltu",		"r,r,la",			"bltu %1,%2,%%b16(%3)",		0, 0, 0 },
  { 0x68000000, 0xfc000000,	"bltu",		"r5:5,r0:5,sb10:16<<2",		0,				0, 0, 0 },
  { 0x0,	0x0,		"bgeu",		"r,r,la",			"bgeu %1,%2,%%b16(%3)",		0, 0, 0 },
  { 0x6c000000, 0xfc000000,	"bgeu",		"r5:5,r0:5,sb10:16<<2",		0,				0, 0, 0 },
  /* Jumps implemented with macros.  */
  { 0x0,	0x0,		"bgt",		"r,r,la",			"blt %2,%1,%%b16(%3)",		0, 0, 0 },
  { 0x0,	0x0,		"bltz",		"r,la",				"blt %1,$r0,%%b16(%2)",		0, 0, 0 },
  { 0x0,	0x0,		"bgtz",		"r,la",				"blt $r0,%1,%%b16(%2)",		0, 0, 0 },
  { 0x0,	0x0,		"ble",		"r,r,la",			"bge %2,%1,%%b16(%3)",		0, 0, 0 },
  { 0x0,	0x0,		"bgez",		"r,la",				"bge %1,$r0,%%b16(%2)",		0, 0, 0 },
  { 0x0,	0x0,		"blez",		"r,la",				"bge $r0,%1,%%b16(%2)",		0, 0, 0 },
  { 0x0,	0x0,		"bgtu",		"r,r,la",			"bltu %2,%1,%%b16(%3)",		0, 0, 0 },
  { 0x0,	0x0,		"bleu",		"r,r,la",			"bgeu %2,%1,%%b16(%3)",		0, 0, 0 },
  { 0x0,	0x0,		"jr",		"r",				"jirl $r0,%1,0",		0, 0, 0 },
  { 0x0,	0x0,		"ret",		"",				"jirl $r0,$r1,0",		0, 0, 0 },
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_lsx_opcodes[] =
{
/* match,	mask,	    name,		format,			macro, include, exclude, pinfo.	*/
  { 0x70000000, 0xffff8000, "vseq.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70008000, 0xffff8000, "vseq.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70010000, 0xffff8000, "vseq.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70018000, 0xffff8000, "vseq.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70020000, 0xffff8000, "vsle.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70028000, 0xffff8000, "vsle.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70030000, 0xffff8000, "vsle.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70038000, 0xffff8000, "vsle.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70040000, 0xffff8000, "vsle.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70048000, 0xffff8000, "vsle.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70050000, 0xffff8000, "vsle.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70058000, 0xffff8000, "vsle.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70060000, 0xffff8000, "vslt.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70068000, 0xffff8000, "vslt.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70070000, 0xffff8000, "vslt.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70078000, 0xffff8000, "vslt.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70080000, 0xffff8000, "vslt.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70088000, 0xffff8000, "vslt.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70090000, 0xffff8000, "vslt.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70098000, 0xffff8000, "vslt.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x700a0000, 0xffff8000, "vadd.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x700a8000, 0xffff8000, "vadd.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x700b0000, 0xffff8000, "vadd.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x700b8000, 0xffff8000, "vadd.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x700c0000, 0xffff8000, "vsub.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x700c8000, 0xffff8000, "vsub.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x700d0000, 0xffff8000, "vsub.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x700d8000, 0xffff8000, "vsub.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70460000, 0xffff8000, "vsadd.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70468000, 0xffff8000, "vsadd.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70470000, 0xffff8000, "vsadd.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70478000, 0xffff8000, "vsadd.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70480000, 0xffff8000, "vssub.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70488000, 0xffff8000, "vssub.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70490000, 0xffff8000, "vssub.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70498000, 0xffff8000, "vssub.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x704a0000, 0xffff8000, "vsadd.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x704a8000, 0xffff8000, "vsadd.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x704b0000, 0xffff8000, "vsadd.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x704b8000, 0xffff8000, "vsadd.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x704c0000, 0xffff8000, "vssub.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x704c8000, 0xffff8000, "vssub.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x704d0000, 0xffff8000, "vssub.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x704d8000, 0xffff8000, "vssub.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70540000, 0xffff8000, "vhaddw.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70548000, 0xffff8000, "vhaddw.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70550000, 0xffff8000, "vhaddw.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70558000, 0xffff8000, "vhaddw.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70560000, 0xffff8000, "vhsubw.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70568000, 0xffff8000, "vhsubw.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70570000, 0xffff8000, "vhsubw.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70578000, 0xffff8000, "vhsubw.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70580000, 0xffff8000, "vhaddw.hu.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70588000, 0xffff8000, "vhaddw.wu.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70590000, 0xffff8000, "vhaddw.du.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70598000, 0xffff8000, "vhaddw.qu.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x705a0000, 0xffff8000, "vhsubw.hu.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x705a8000, 0xffff8000, "vhsubw.wu.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x705b0000, 0xffff8000, "vhsubw.du.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x705b8000, 0xffff8000, "vhsubw.qu.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x705c0000, 0xffff8000, "vadda.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x705c8000, 0xffff8000, "vadda.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x705d0000, 0xffff8000, "vadda.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x705d8000, 0xffff8000, "vadda.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70600000, 0xffff8000, "vabsd.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70608000, 0xffff8000, "vabsd.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70610000, 0xffff8000, "vabsd.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70618000, 0xffff8000, "vabsd.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70620000, 0xffff8000, "vabsd.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70628000, 0xffff8000, "vabsd.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70630000, 0xffff8000, "vabsd.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70638000, 0xffff8000, "vabsd.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70640000, 0xffff8000, "vavg.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70648000, 0xffff8000, "vavg.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70650000, 0xffff8000, "vavg.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70658000, 0xffff8000, "vavg.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70660000, 0xffff8000, "vavg.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70668000, 0xffff8000, "vavg.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70670000, 0xffff8000, "vavg.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70678000, 0xffff8000, "vavg.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70680000, 0xffff8000, "vavgr.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70688000, 0xffff8000, "vavgr.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70690000, 0xffff8000, "vavgr.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70698000, 0xffff8000, "vavgr.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x706a0000, 0xffff8000, "vavgr.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x706a8000, 0xffff8000, "vavgr.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x706b0000, 0xffff8000, "vavgr.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x706b8000, 0xffff8000, "vavgr.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70700000, 0xffff8000, "vmax.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70708000, 0xffff8000, "vmax.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70710000, 0xffff8000, "vmax.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70718000, 0xffff8000, "vmax.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70720000, 0xffff8000, "vmin.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70728000, 0xffff8000, "vmin.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70730000, 0xffff8000, "vmin.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70738000, 0xffff8000, "vmin.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70740000, 0xffff8000, "vmax.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70748000, 0xffff8000, "vmax.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70750000, 0xffff8000, "vmax.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70758000, 0xffff8000, "vmax.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70760000, 0xffff8000, "vmin.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70768000, 0xffff8000, "vmin.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70770000, 0xffff8000, "vmin.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70778000, 0xffff8000, "vmin.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70840000, 0xffff8000, "vmul.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70848000, 0xffff8000, "vmul.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70850000, 0xffff8000, "vmul.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70858000, 0xffff8000, "vmul.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70860000, 0xffff8000, "vmuh.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70868000, 0xffff8000, "vmuh.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70870000, 0xffff8000, "vmuh.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70878000, 0xffff8000, "vmuh.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70880000, 0xffff8000, "vmuh.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70888000, 0xffff8000, "vmuh.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70890000, 0xffff8000, "vmuh.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70898000, 0xffff8000, "vmuh.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a80000, 0xffff8000, "vmadd.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a88000, 0xffff8000, "vmadd.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a90000, 0xffff8000, "vmadd.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a98000, 0xffff8000, "vmadd.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70aa0000, 0xffff8000, "vmsub.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70aa8000, 0xffff8000, "vmsub.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ab0000, 0xffff8000, "vmsub.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ab8000, 0xffff8000, "vmsub.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e00000, 0xffff8000, "vdiv.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e08000, 0xffff8000, "vdiv.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e10000, 0xffff8000, "vdiv.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e18000, 0xffff8000, "vdiv.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e20000, 0xffff8000, "vmod.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e28000, 0xffff8000, "vmod.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e30000, 0xffff8000, "vmod.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e38000, 0xffff8000, "vmod.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e40000, 0xffff8000, "vdiv.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e48000, 0xffff8000, "vdiv.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e50000, 0xffff8000, "vdiv.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e58000, 0xffff8000, "vdiv.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e60000, 0xffff8000, "vmod.bu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e68000, 0xffff8000, "vmod.hu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e70000, 0xffff8000, "vmod.wu",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e78000, 0xffff8000, "vmod.du",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e80000, 0xffff8000, "vsll.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e88000, 0xffff8000, "vsll.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e90000, 0xffff8000, "vsll.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70e98000, 0xffff8000, "vsll.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ea0000, 0xffff8000, "vsrl.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ea8000, 0xffff8000, "vsrl.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70eb0000, 0xffff8000, "vsrl.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70eb8000, 0xffff8000, "vsrl.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ec0000, 0xffff8000, "vsra.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ec8000, 0xffff8000, "vsra.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ed0000, 0xffff8000, "vsra.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ed8000, 0xffff8000, "vsra.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ee0000, 0xffff8000, "vrotr.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ee8000, 0xffff8000, "vrotr.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ef0000, 0xffff8000, "vrotr.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ef8000, 0xffff8000, "vrotr.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f00000, 0xffff8000, "vsrlr.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f08000, 0xffff8000, "vsrlr.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f10000, 0xffff8000, "vsrlr.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f18000, 0xffff8000, "vsrlr.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f20000, 0xffff8000, "vsrar.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f28000, 0xffff8000, "vsrar.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f30000, 0xffff8000, "vsrar.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f38000, 0xffff8000, "vsrar.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f48000, 0xffff8000, "vsrln.b.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f50000, 0xffff8000, "vsrln.h.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f58000, 0xffff8000, "vsrln.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f68000, 0xffff8000, "vsran.b.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f70000, 0xffff8000, "vsran.h.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f78000, 0xffff8000, "vsran.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f88000, 0xffff8000, "vsrlrn.b.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f90000, 0xffff8000, "vsrlrn.h.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70f98000, 0xffff8000, "vsrlrn.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70fa8000, 0xffff8000, "vsrarn.b.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70fb0000, 0xffff8000, "vsrarn.h.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70fb8000, 0xffff8000, "vsrarn.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70fc8000, 0xffff8000, "vssrln.b.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70fd0000, 0xffff8000, "vssrln.h.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70fd8000, 0xffff8000, "vssrln.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70fe8000, 0xffff8000, "vssran.b.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ff0000, 0xffff8000, "vssran.h.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ff8000, 0xffff8000, "vssran.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71008000, 0xffff8000, "vssrlrn.b.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71010000, 0xffff8000, "vssrlrn.h.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71018000, 0xffff8000, "vssrlrn.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71028000, 0xffff8000, "vssrarn.b.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71030000, 0xffff8000, "vssrarn.h.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71038000, 0xffff8000, "vssrarn.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71048000, 0xffff8000, "vssrln.bu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71050000, 0xffff8000, "vssrln.hu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71058000, 0xffff8000, "vssrln.wu.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71068000, 0xffff8000, "vssran.bu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71070000, 0xffff8000, "vssran.hu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71078000, 0xffff8000, "vssran.wu.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71088000, 0xffff8000, "vssrlrn.bu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71090000, 0xffff8000, "vssrlrn.hu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71098000, 0xffff8000, "vssrlrn.wu.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710a8000, 0xffff8000, "vssrarn.bu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710b0000, 0xffff8000, "vssrarn.hu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710b8000, 0xffff8000, "vssrarn.wu.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710c0000, 0xffff8000, "vbitclr.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710c8000, 0xffff8000, "vbitclr.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710d0000, 0xffff8000, "vbitclr.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710d8000, 0xffff8000, "vbitclr.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710e0000, 0xffff8000, "vbitset.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710e8000, 0xffff8000, "vbitset.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710f0000, 0xffff8000, "vbitset.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x710f8000, 0xffff8000, "vbitset.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71100000, 0xffff8000, "vbitrev.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71108000, 0xffff8000, "vbitrev.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71110000, 0xffff8000, "vbitrev.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71118000, 0xffff8000, "vbitrev.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71160000, 0xffff8000, "vpackev.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71168000, 0xffff8000, "vpackev.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71170000, 0xffff8000, "vpackev.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71178000, 0xffff8000, "vpackev.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71180000, 0xffff8000, "vpackod.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71188000, 0xffff8000, "vpackod.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71190000, 0xffff8000, "vpackod.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71198000, 0xffff8000, "vpackod.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711a0000, 0xffff8000, "vilvl.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711a8000, 0xffff8000, "vilvl.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711b0000, 0xffff8000, "vilvl.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711b8000, 0xffff8000, "vilvl.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711c0000, 0xffff8000, "vilvh.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711c8000, 0xffff8000, "vilvh.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711d0000, 0xffff8000, "vilvh.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711d8000, 0xffff8000, "vilvh.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711e0000, 0xffff8000, "vpickev.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711e8000, 0xffff8000, "vpickev.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711f0000, 0xffff8000, "vpickev.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x711f8000, 0xffff8000, "vpickev.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71200000, 0xffff8000, "vpickod.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71208000, 0xffff8000, "vpickod.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71210000, 0xffff8000, "vpickod.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71218000, 0xffff8000, "vpickod.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71220000, 0xffff8000, "vreplve.b",	"v0:5,v5:5,r10:5",	0, 0, 0, 0},
  { 0x71228000, 0xffff8000, "vreplve.h",	"v0:5,v5:5,r10:5",	0, 0, 0, 0},
  { 0x71230000, 0xffff8000, "vreplve.w",	"v0:5,v5:5,r10:5",	0, 0, 0, 0},
  { 0x71238000, 0xffff8000, "vreplve.d",	"v0:5,v5:5,r10:5",	0, 0, 0, 0},
  { 0x71260000, 0xffff8000, "vand.v",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71268000, 0xffff8000, "vor.v",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71270000, 0xffff8000, "vxor.v",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71278000, 0xffff8000, "vnor.v",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71280000, 0xffff8000, "vandn.v",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71288000, 0xffff8000, "vorn.v",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x712b0000, 0xffff8000, "vfrstp.b",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x712b8000, 0xffff8000, "vfrstp.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x712d0000, 0xffff8000, "vadd.q",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x712d8000, 0xffff8000, "vsub.q",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x712e0000, 0xffff8000, "vsigncov.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x712e8000, 0xffff8000, "vsigncov.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x712f0000, 0xffff8000, "vsigncov.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x712f8000, 0xffff8000, "vsigncov.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71308000, 0xffff8000, "vfadd.s",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71310000, 0xffff8000, "vfadd.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71328000, 0xffff8000, "vfsub.s",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71330000, 0xffff8000, "vfsub.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71388000, 0xffff8000, "vfmul.s",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71390000, 0xffff8000, "vfmul.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x713a8000, 0xffff8000, "vfdiv.s",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x713b0000, 0xffff8000, "vfdiv.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x713c8000, 0xffff8000, "vfmax.s",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x713d0000, 0xffff8000, "vfmax.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x713e8000, 0xffff8000, "vfmin.s",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x713f0000, 0xffff8000, "vfmin.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71408000, 0xffff8000, "vfmaxa.s",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71410000, 0xffff8000, "vfmaxa.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71428000, 0xffff8000, "vfmina.s",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71430000, 0xffff8000, "vfmina.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71460000, 0xffff8000, "vfcvt.h.s",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71468000, 0xffff8000, "vfcvt.s.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71480000, 0xffff8000, "vffint.s.l",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x71498000, 0xffff8000, "vftint.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x714a0000, 0xffff8000, "vftintrm.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x714a8000, 0xffff8000, "vftintrp.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x714b0000, 0xffff8000, "vftintrz.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x714b8000, 0xffff8000, "vftintrne.w.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x717a8000, 0xffff8000, "vshuf.h",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x717b0000, 0xffff8000, "vshuf.w",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x717b8000, 0xffff8000, "vshuf.d",		"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x72800000, 0xffff8000, "vseqi.b",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72808000, 0xffff8000, "vseqi.h",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72810000, 0xffff8000, "vseqi.w",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72818000, 0xffff8000, "vseqi.d",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72820000, 0xffff8000, "vslei.b",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72828000, 0xffff8000, "vslei.h",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72830000, 0xffff8000, "vslei.w",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72838000, 0xffff8000, "vslei.d",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72840000, 0xffff8000, "vslei.bu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72848000, 0xffff8000, "vslei.hu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72850000, 0xffff8000, "vslei.wu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72858000, 0xffff8000, "vslei.du",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72860000, 0xffff8000, "vslti.b",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72868000, 0xffff8000, "vslti.h",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72870000, 0xffff8000, "vslti.w",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72878000, 0xffff8000, "vslti.d",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72880000, 0xffff8000, "vslti.bu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72888000, 0xffff8000, "vslti.hu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72890000, 0xffff8000, "vslti.wu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72898000, 0xffff8000, "vslti.du",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728a0000, 0xffff8000, "vaddi.bu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728a8000, 0xffff8000, "vaddi.hu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728b0000, 0xffff8000, "vaddi.wu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728b8000, 0xffff8000, "vaddi.du",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728c0000, 0xffff8000, "vsubi.bu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728c8000, 0xffff8000, "vsubi.hu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728d0000, 0xffff8000, "vsubi.wu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728d8000, 0xffff8000, "vsubi.du",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728e0000, 0xffff8000, "vbsll.v",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x728e8000, 0xffff8000, "vbsrl.v",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72900000, 0xffff8000, "vmaxi.b",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72908000, 0xffff8000, "vmaxi.h",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72910000, 0xffff8000, "vmaxi.w",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72918000, 0xffff8000, "vmaxi.d",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72920000, 0xffff8000, "vmini.b",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72928000, 0xffff8000, "vmini.h",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72930000, 0xffff8000, "vmini.w",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72938000, 0xffff8000, "vmini.d",		"v0:5,v5:5,s10:5",	0, 0, 0, 0},
  { 0x72940000, 0xffff8000, "vmaxi.bu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72948000, 0xffff8000, "vmaxi.hu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72950000, 0xffff8000, "vmaxi.wu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72958000, 0xffff8000, "vmaxi.du",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72960000, 0xffff8000, "vmini.bu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72968000, 0xffff8000, "vmini.hu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72970000, 0xffff8000, "vmini.wu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72978000, 0xffff8000, "vmini.du",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x729a0000, 0xffff8000, "vfrstpi.b",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x729a8000, 0xffff8000, "vfrstpi.h",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x729c0000, 0xfffffc00, "vclo.b",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c0400, 0xfffffc00, "vclo.h",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c0800, 0xfffffc00, "vclo.w",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c0c00, 0xfffffc00, "vclo.d",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c1000, 0xfffffc00, "vclz.b",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c1400, 0xfffffc00, "vclz.h",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c1800, 0xfffffc00, "vclz.w",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c1c00, 0xfffffc00, "vclz.d",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c2000, 0xfffffc00, "vpcnt.b",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c2400, 0xfffffc00, "vpcnt.h",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c2800, 0xfffffc00, "vpcnt.w",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c2c00, 0xfffffc00, "vpcnt.d",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c3000, 0xfffffc00, "vneg.b",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c3400, 0xfffffc00, "vneg.h",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c3800, 0xfffffc00, "vneg.w",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c3c00, 0xfffffc00, "vneg.d",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c4000, 0xfffffc00, "vmskltz.b",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c4400, 0xfffffc00, "vmskltz.h",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c4800, 0xfffffc00, "vmskltz.w",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c4c00, 0xfffffc00, "vmskltz.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c5000, 0xfffffc00, "vmskgez.b",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c6000, 0xfffffc00, "vmsknz.b",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729c9800, 0xfffffc18, "vseteqz.v",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729c9c00, 0xfffffc18, "vsetnez.v",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729ca000, 0xfffffc18, "vsetanyeqz.b",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729ca400, 0xfffffc18, "vsetanyeqz.h",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729ca800, 0xfffffc18, "vsetanyeqz.w",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729cac00, 0xfffffc18, "vsetanyeqz.d",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729cb000, 0xfffffc18, "vsetallnez.b",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729cb400, 0xfffffc18, "vsetallnez.h",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729cb800, 0xfffffc18, "vsetallnez.w",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729cbc00, 0xfffffc18, "vsetallnez.d",	"c0:3,v5:5",		0, 0, 0, 0},
  { 0x729cc400, 0xfffffc00, "vflogb.s",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729cc800, 0xfffffc00, "vflogb.d",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729cd400, 0xfffffc00, "vfclass.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729cd800, 0xfffffc00, "vfclass.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ce400, 0xfffffc00, "vfsqrt.s",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ce800, 0xfffffc00, "vfsqrt.d",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729cf400, 0xfffffc00, "vfrecip.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729cf800, 0xfffffc00, "vfrecip.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d0400, 0xfffffc00, "vfrsqrt.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d0800, 0xfffffc00, "vfrsqrt.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d1400, 0xfffffc00, "vfrecipe.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d1800, 0xfffffc00, "vfrecipe.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d2400, 0xfffffc00, "vfrsqrte.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d2800, 0xfffffc00, "vfrsqrte.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d3400, 0xfffffc00, "vfrint.s",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d3800, 0xfffffc00, "vfrint.d",		"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d4400, 0xfffffc00, "vfrintrm.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d4800, 0xfffffc00, "vfrintrm.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d5400, 0xfffffc00, "vfrintrp.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d5800, 0xfffffc00, "vfrintrp.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d6400, 0xfffffc00, "vfrintrz.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d6800, 0xfffffc00, "vfrintrz.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d7400, 0xfffffc00, "vfrintrne.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729d7800, 0xfffffc00, "vfrintrne.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729de800, 0xfffffc00, "vfcvtl.s.h",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729dec00, 0xfffffc00, "vfcvth.s.h",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729df000, 0xfffffc00, "vfcvtl.d.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729df400, 0xfffffc00, "vfcvth.d.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e0000, 0xfffffc00, "vffint.s.w",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e0400, 0xfffffc00, "vffint.s.wu",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e0800, 0xfffffc00, "vffint.d.l",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e0c00, 0xfffffc00, "vffint.d.lu",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e1000, 0xfffffc00, "vffintl.d.w",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e1400, 0xfffffc00, "vffinth.d.w",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e3000, 0xfffffc00, "vftint.w.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e3400, 0xfffffc00, "vftint.l.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e3800, 0xfffffc00, "vftintrm.w.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e3c00, 0xfffffc00, "vftintrm.l.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e4000, 0xfffffc00, "vftintrp.w.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e4400, 0xfffffc00, "vftintrp.l.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e4800, 0xfffffc00, "vftintrz.w.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e4c00, 0xfffffc00, "vftintrz.l.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e5000, 0xfffffc00, "vftintrne.w.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e5400, 0xfffffc00, "vftintrne.l.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e5800, 0xfffffc00, "vftint.wu.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e5c00, 0xfffffc00, "vftint.lu.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e7000, 0xfffffc00, "vftintrz.wu.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e7400, 0xfffffc00, "vftintrz.lu.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e8000, 0xfffffc00, "vftintl.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e8400, 0xfffffc00, "vftinth.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e8800, 0xfffffc00, "vftintrml.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e8c00, 0xfffffc00, "vftintrmh.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e9000, 0xfffffc00, "vftintrpl.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e9400, 0xfffffc00, "vftintrph.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e9800, 0xfffffc00, "vftintrzl.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729e9c00, 0xfffffc00, "vftintrzh.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ea000, 0xfffffc00, "vftintrnel.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ea400, 0xfffffc00, "vftintrneh.l.s",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ee000, 0xfffffc00, "vexth.h.b",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ee400, 0xfffffc00, "vexth.w.h",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ee800, 0xfffffc00, "vexth.d.w",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729eec00, 0xfffffc00, "vexth.q.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ef000, 0xfffffc00, "vexth.hu.bu",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ef400, 0xfffffc00, "vexth.wu.hu",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729ef800, 0xfffffc00, "vexth.du.wu",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729efc00, 0xfffffc00, "vexth.qu.du",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x729f0000, 0xfffffc00, "vreplgr2vr.b",	"v0:5,r5:5",		0, 0, 0, 0},
  { 0x729f0400, 0xfffffc00, "vreplgr2vr.h",	"v0:5,r5:5",		0, 0, 0, 0},
  { 0x729f0800, 0xfffffc00, "vreplgr2vr.w",	"v0:5,r5:5",		0, 0, 0, 0},
  { 0x729f0c00, 0xfffffc00, "vreplgr2vr.d",	"v0:5,r5:5",		0, 0, 0, 0},
  { 0x72a02000, 0xffffe000, "vrotri.b",		"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x72a04000, 0xffffc000, "vrotri.h",		"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x72a08000, 0xffff8000, "vrotri.w",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72a10000, 0xffff0000, "vrotri.d",		"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x72a42000, 0xffffe000, "vsrlri.b",		"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x72a44000, 0xffffc000, "vsrlri.h",		"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x72a48000, 0xffff8000, "vsrlri.w",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72a50000, 0xffff0000, "vsrlri.d",		"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x72a82000, 0xffffe000, "vsrari.b",		"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x72a84000, 0xffffc000, "vsrari.h",		"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x72a88000, 0xffff8000, "vsrari.w",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x72a90000, 0xffff0000, "vsrari.d",		"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x72eb8000, 0xffffc000, "vinsgr2vr.b",	"v0:5,r5:5,u10:4",	0, 0, 0, 0},
  { 0x72ebc000, 0xffffe000, "vinsgr2vr.h",	"v0:5,r5:5,u10:3",	0, 0, 0, 0},
  { 0x72ebe000, 0xfffff000, "vinsgr2vr.w",	"v0:5,r5:5,u10:2",	0, 0, 0, 0},
  { 0x72ebf000, 0xfffff800, "vinsgr2vr.d",	"v0:5,r5:5,u10:1",	0, 0, 0, 0},
  { 0x72ef8000, 0xffffc000, "vpickve2gr.b",	"r0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x72efc000, 0xffffe000, "vpickve2gr.h",	"r0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x72efe000, 0xfffff000, "vpickve2gr.w",	"r0:5,v5:5,u10:2",	0, 0, 0, 0},
  { 0x72eff000, 0xfffff800, "vpickve2gr.d",	"r0:5,v5:5,u10:1",	0, 0, 0, 0},
  { 0x72f38000, 0xffffc000, "vpickve2gr.bu",	"r0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x72f3c000, 0xffffe000, "vpickve2gr.hu",	"r0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x72f3e000, 0xfffff000, "vpickve2gr.wu",	"r0:5,v5:5,u10:2",	0, 0, 0, 0},
  { 0x72f3f000, 0xfffff800, "vpickve2gr.du",	"r0:5,v5:5,u10:1",	0, 0, 0, 0},
  { 0x72f78000, 0xffffc000, "vreplvei.b",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x72f7c000, 0xffffe000, "vreplvei.h",	"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x72f7e000, 0xfffff000, "vreplvei.w",	"v0:5,v5:5,u10:2",	0, 0, 0, 0},
  { 0x72f7f000, 0xfffff800, "vreplvei.d",	"v0:5,v5:5,u10:1",	0, 0, 0, 0},
  { 0x73082000, 0xffffe000, "vsllwil.h.b",	"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x73084000, 0xffffc000, "vsllwil.w.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73088000, 0xffff8000, "vsllwil.d.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73090000, 0xfffffc00, "vextl.q.d",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x730c2000, 0xffffe000, "vsllwil.hu.bu",	"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x730c4000, 0xffffc000, "vsllwil.wu.hu",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x730c8000, 0xffff8000, "vsllwil.du.wu",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x730d0000, 0xfffffc00, "vextl.qu.du",	"v0:5,v5:5",		0, 0, 0, 0},
  { 0x73102000, 0xffffe000, "vbitclri.b",	"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x73104000, 0xffffc000, "vbitclri.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73108000, 0xffff8000, "vbitclri.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73110000, 0xffff0000, "vbitclri.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73142000, 0xffffe000, "vbitseti.b",	"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x73144000, 0xffffc000, "vbitseti.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73148000, 0xffff8000, "vbitseti.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73150000, 0xffff0000, "vbitseti.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73182000, 0xffffe000, "vbitrevi.b",	"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x73184000, 0xffffc000, "vbitrevi.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73188000, 0xffff8000, "vbitrevi.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73190000, 0xffff0000, "vbitrevi.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73242000, 0xffffe000, "vsat.b",		"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x73244000, 0xffffc000, "vsat.h",		"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73248000, 0xffff8000, "vsat.w",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73250000, 0xffff0000, "vsat.d",		"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73282000, 0xffffe000, "vsat.bu",		"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x73284000, 0xffffc000, "vsat.hu",		"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73288000, 0xffff8000, "vsat.wu",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73290000, 0xffff0000, "vsat.du",		"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x732c2000, 0xffffe000, "vslli.b",		"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x732c4000, 0xffffc000, "vslli.h",		"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x732c8000, 0xffff8000, "vslli.w",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x732d0000, 0xffff0000, "vslli.d",		"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73302000, 0xffffe000, "vsrli.b",		"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x73304000, 0xffffc000, "vsrli.h",		"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73308000, 0xffff8000, "vsrli.w",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73310000, 0xffff0000, "vsrli.d",		"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73342000, 0xffffe000, "vsrai.b",		"v0:5,v5:5,u10:3",	0, 0, 0, 0},
  { 0x73344000, 0xffffc000, "vsrai.h",		"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73348000, 0xffff8000, "vsrai.w",		"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73350000, 0xffff0000, "vsrai.d",		"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73404000, 0xffffc000, "vsrlni.b.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73408000, 0xffff8000, "vsrlni.h.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73410000, 0xffff0000, "vsrlni.w.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73420000, 0xfffe0000, "vsrlni.d.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x73484000, 0xffffc000, "vssrlni.b.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73488000, 0xffff8000, "vssrlni.h.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73490000, 0xffff0000, "vssrlni.w.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x734a0000, 0xfffe0000, "vssrlni.d.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x73444000, 0xffffc000, "vsrlrni.b.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73448000, 0xffff8000, "vsrlrni.h.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73450000, 0xffff0000, "vsrlrni.w.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73460000, 0xfffe0000, "vsrlrni.d.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x734c4000, 0xffffc000, "vssrlni.bu.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x734c8000, 0xffff8000, "vssrlni.hu.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x734d0000, 0xffff0000, "vssrlni.wu.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x734e0000, 0xfffe0000, "vssrlni.du.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x73504000, 0xffffc000, "vssrlrni.b.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73508000, 0xffff8000, "vssrlrni.h.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73510000, 0xffff0000, "vssrlrni.w.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73520000, 0xfffe0000, "vssrlrni.d.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x73544000, 0xffffc000, "vssrlrni.bu.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73548000, 0xffff8000, "vssrlrni.hu.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73550000, 0xffff0000, "vssrlrni.wu.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73560000, 0xfffe0000, "vssrlrni.du.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x73584000, 0xffffc000, "vsrani.b.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73588000, 0xffff8000, "vsrani.h.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73590000, 0xffff0000, "vsrani.w.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x735a0000, 0xfffe0000, "vsrani.d.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x735c4000, 0xffffc000, "vsrarni.b.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x735c8000, 0xffff8000, "vsrarni.h.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x735d0000, 0xffff0000, "vsrarni.w.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x735e0000, 0xfffe0000, "vsrarni.d.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x73604000, 0xffffc000, "vssrani.b.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73608000, 0xffff8000, "vssrani.h.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73610000, 0xffff0000, "vssrani.w.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73620000, 0xfffe0000, "vssrani.d.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x73644000, 0xffffc000, "vssrani.bu.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73648000, 0xffff8000, "vssrani.hu.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73650000, 0xffff0000, "vssrani.wu.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x73660000, 0xfffe0000, "vssrani.du.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x73684000, 0xffffc000, "vssrarni.b.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x73688000, 0xffff8000, "vssrarni.h.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x73690000, 0xffff0000, "vssrarni.w.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x736a0000, 0xfffe0000, "vssrarni.d.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x736c4000, 0xffffc000, "vssrarni.bu.h",	"v0:5,v5:5,u10:4",	0, 0, 0, 0},
  { 0x736c8000, 0xffff8000, "vssrarni.hu.w",	"v0:5,v5:5,u10:5",	0, 0, 0, 0},
  { 0x736d0000, 0xffff0000, "vssrarni.wu.d",	"v0:5,v5:5,u10:6",	0, 0, 0, 0},
  { 0x736e0000, 0xfffe0000, "vssrarni.du.q",	"v0:5,v5:5,u10:7",	0, 0, 0, 0},
  { 0x73800000, 0xfffc0000, "vextrins.d",	"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73840000, 0xfffc0000, "vextrins.w",	"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73880000, 0xfffc0000, "vextrins.h",	"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x738c0000, 0xfffc0000, "vextrins.b",	"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73900000, 0xfffc0000, "vshuf4i.b",	"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73940000, 0xfffc0000, "vshuf4i.h",	"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73980000, 0xfffc0000, "vshuf4i.w",	"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x739c0000, 0xfffc0000, "vshuf4i.d",	"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73c40000, 0xfffc0000, "vbitseli.b",	"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73d00000, 0xfffc0000, "vandi.b",		"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73d40000, 0xfffc0000, "vori.b",		"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73d80000, 0xfffc0000, "vxori.b",		"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0x73dc0000, 0xfffc0000, "vnori.b",		"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0,		0,	    "vrepli.b",		"v,s0:10",		"vldi %1,(%2)&0x3ff", 0, 0, 0},
  { 0x701e0000, 0xffff8000, "vaddwev.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x701e8000, 0xffff8000, "vaddwev.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x701f0000, 0xffff8000, "vaddwev.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x701f8000, 0xffff8000, "vaddwev.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x702e0000, 0xffff8000, "vaddwev.h.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x702e8000, 0xffff8000, "vaddwev.w.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x702f0000, 0xffff8000, "vaddwev.d.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x702f8000, 0xffff8000, "vaddwev.q.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x703e0000, 0xffff8000, "vaddwev.h.bu.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x703e8000, 0xffff8000, "vaddwev.w.hu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x703f0000, 0xffff8000, "vaddwev.d.wu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x703f8000, 0xffff8000, "vaddwev.q.du.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70220000, 0xffff8000, "vaddwod.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70228000, 0xffff8000, "vaddwod.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70230000, 0xffff8000, "vaddwod.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70238000, 0xffff8000, "vaddwod.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70320000, 0xffff8000, "vaddwod.h.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70328000, 0xffff8000, "vaddwod.w.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70330000, 0xffff8000, "vaddwod.d.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70338000, 0xffff8000, "vaddwod.q.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70400000, 0xffff8000, "vaddwod.h.bu.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70408000, 0xffff8000, "vaddwod.w.hu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70410000, 0xffff8000, "vaddwod.d.wu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70418000, 0xffff8000, "vaddwod.q.du.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ac0000, 0xffff8000, "vmaddwev.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ac8000, 0xffff8000, "vmaddwev.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ad0000, 0xffff8000, "vmaddwev.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ad8000, 0xffff8000, "vmaddwev.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70b40000, 0xffff8000, "vmaddwev.h.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70b48000, 0xffff8000, "vmaddwev.w.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70b50000, 0xffff8000, "vmaddwev.d.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70b58000, 0xffff8000, "vmaddwev.q.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70bc0000, 0xffff8000, "vmaddwev.h.bu.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70bc8000, 0xffff8000, "vmaddwev.w.hu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70bd0000, 0xffff8000, "vmaddwev.d.wu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70bd8000, 0xffff8000, "vmaddwev.q.du.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ae0000, 0xffff8000, "vmaddwod.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70ae8000, 0xffff8000, "vmaddwod.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70af0000, 0xffff8000, "vmaddwod.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70af8000, 0xffff8000, "vmaddwod.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70b60000, 0xffff8000, "vmaddwod.h.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70b68000, 0xffff8000, "vmaddwod.w.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70b70000, 0xffff8000, "vmaddwod.d.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70b78000, 0xffff8000, "vmaddwod.q.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70be0000, 0xffff8000, "vmaddwod.h.bu.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70be8000, 0xffff8000, "vmaddwod.w.hu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70bf0000, 0xffff8000, "vmaddwod.d.wu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70bf8000, 0xffff8000, "vmaddwod.q.du.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70900000, 0xffff8000, "vmulwev.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70908000, 0xffff8000, "vmulwev.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70910000, 0xffff8000, "vmulwev.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70918000, 0xffff8000, "vmulwev.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70980000, 0xffff8000, "vmulwev.h.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70988000, 0xffff8000, "vmulwev.w.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70990000, 0xffff8000, "vmulwev.d.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70998000, 0xffff8000, "vmulwev.q.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a00000, 0xffff8000, "vmulwev.h.bu.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a08000, 0xffff8000, "vmulwev.w.hu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a10000, 0xffff8000, "vmulwev.d.wu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a18000, 0xffff8000, "vmulwev.q.du.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70920000, 0xffff8000, "vmulwod.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70928000, 0xffff8000, "vmulwod.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70930000, 0xffff8000, "vmulwod.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70938000, 0xffff8000, "vmulwod.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x709a0000, 0xffff8000, "vmulwod.h.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x709a8000, 0xffff8000, "vmulwod.w.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x709b0000, 0xffff8000, "vmulwod.d.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x709b8000, 0xffff8000, "vmulwod.q.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a20000, 0xffff8000, "vmulwod.h.bu.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a28000, 0xffff8000, "vmulwod.w.hu.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a30000, 0xffff8000, "vmulwod.d.wu.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70a38000, 0xffff8000, "vmulwod.q.du.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70200000, 0xffff8000, "vsubwev.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70208000, 0xffff8000, "vsubwev.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70210000, 0xffff8000, "vsubwev.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70218000, 0xffff8000, "vsubwev.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70300000, 0xffff8000, "vsubwev.h.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70308000, 0xffff8000, "vsubwev.w.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70310000, 0xffff8000, "vsubwev.d.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70318000, 0xffff8000, "vsubwev.q.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70240000, 0xffff8000, "vsubwod.h.b",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70248000, 0xffff8000, "vsubwod.w.h",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70250000, 0xffff8000, "vsubwod.d.w",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70258000, 0xffff8000, "vsubwod.q.d",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70340000, 0xffff8000, "vsubwod.h.bu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70348000, 0xffff8000, "vsubwod.w.hu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70350000, 0xffff8000, "vsubwod.d.wu",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0x70358000, 0xffff8000, "vsubwod.q.du",	"v0:5,v5:5,v10:5",	0, 0, 0, 0 },
  { 0,		0,	    "vrepli.d",		"v,s0:10",		"vldi %1,((%2)&0x3ff)|0xc00", 0, 0, 0},
  { 0,		0,	    "vrepli.h",		"v,s0:10",		"vldi %1,((%2)&0x3ff)|0x400", 0, 0, 0},
  { 0,		0,	    "vrepli.w",		"v,s0:10",		"vldi %1,((%2)&0x3ff)|0x800", 0, 0, 0},
  { 0x73e00000, 0xfffc0000, "vldi",		"v0:5,s5:13",		0, 0, 0, 0},
  { 0x73e40000, 0xfffc0000, "vpermi.w",		"v0:5,v5:5,u10:8",	0, 0, 0, 0},
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_lasx_opcodes[] =
{
/* match,    mask,		name, format, macro, include, exclude, pinfo.	*/
  { 0x74000000, 0xffff8000, "xvseq.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74008000, 0xffff8000, "xvseq.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74010000, 0xffff8000, "xvseq.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74018000, 0xffff8000, "xvseq.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74020000, 0xffff8000, "xvsle.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74028000, 0xffff8000, "xvsle.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74030000, 0xffff8000, "xvsle.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74038000, 0xffff8000, "xvsle.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74040000, 0xffff8000, "xvsle.bu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74048000, 0xffff8000, "xvsle.hu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74050000, 0xffff8000, "xvsle.wu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74058000, 0xffff8000, "xvsle.du",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74060000, 0xffff8000, "xvslt.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74068000, 0xffff8000, "xvslt.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74070000, 0xffff8000, "xvslt.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74078000, 0xffff8000, "xvslt.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74080000, 0xffff8000, "xvslt.bu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74088000, 0xffff8000, "xvslt.hu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74090000, 0xffff8000, "xvslt.wu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74098000, 0xffff8000, "xvslt.du",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x740a0000, 0xffff8000, "xvadd.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x740a8000, 0xffff8000, "xvadd.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x740b0000, 0xffff8000, "xvadd.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x740b8000, 0xffff8000, "xvadd.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x740c0000, 0xffff8000, "xvsub.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x740c8000, 0xffff8000, "xvsub.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x740d0000, 0xffff8000, "xvsub.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x740d8000, 0xffff8000, "xvsub.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74460000, 0xffff8000, "xvsadd.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74468000, 0xffff8000, "xvsadd.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74470000, 0xffff8000, "xvsadd.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74478000, 0xffff8000, "xvsadd.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74480000, 0xffff8000, "xvssub.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74488000, 0xffff8000, "xvssub.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74490000, 0xffff8000, "xvssub.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74498000, 0xffff8000, "xvssub.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x744a0000, 0xffff8000, "xvsadd.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x744a8000, 0xffff8000, "xvsadd.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x744b0000, 0xffff8000, "xvsadd.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x744b8000, 0xffff8000, "xvsadd.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x744c0000, 0xffff8000, "xvssub.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x744c8000, 0xffff8000, "xvssub.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x744d0000, 0xffff8000, "xvssub.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x744d8000, 0xffff8000, "xvssub.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74540000, 0xffff8000, "xvhaddw.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74548000, 0xffff8000, "xvhaddw.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74550000, 0xffff8000, "xvhaddw.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74558000, 0xffff8000, "xvhaddw.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74560000, 0xffff8000, "xvhsubw.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74568000, 0xffff8000, "xvhsubw.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74570000, 0xffff8000, "xvhsubw.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74578000, 0xffff8000, "xvhsubw.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74580000, 0xffff8000, "xvhaddw.hu.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74588000, 0xffff8000, "xvhaddw.wu.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74590000, 0xffff8000, "xvhaddw.du.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74598000, 0xffff8000, "xvhaddw.qu.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x745a0000, 0xffff8000, "xvhsubw.hu.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x745a8000, 0xffff8000, "xvhsubw.wu.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x745b0000, 0xffff8000, "xvhsubw.du.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x745b8000, 0xffff8000, "xvhsubw.qu.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x741e0000, 0xffff8000, "xvaddwev.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x741e8000, 0xffff8000, "xvaddwev.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x741f0000, 0xffff8000, "xvaddwev.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x741f8000, 0xffff8000, "xvaddwev.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x742e0000, 0xffff8000, "xvaddwev.h.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x742e8000, 0xffff8000, "xvaddwev.w.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x742f0000, 0xffff8000, "xvaddwev.d.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x742f8000, 0xffff8000, "xvaddwev.q.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x743e0000, 0xffff8000, "xvaddwev.h.bu.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x743e8000, 0xffff8000, "xvaddwev.w.hu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x743f0000, 0xffff8000, "xvaddwev.d.wu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x743f8000, 0xffff8000, "xvaddwev.q.du.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74220000, 0xffff8000, "xvaddwod.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74228000, 0xffff8000, "xvaddwod.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74230000, 0xffff8000, "xvaddwod.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74238000, 0xffff8000, "xvaddwod.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74320000, 0xffff8000, "xvaddwod.h.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74328000, 0xffff8000, "xvaddwod.w.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74330000, 0xffff8000, "xvaddwod.d.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74338000, 0xffff8000, "xvaddwod.q.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74400000, 0xffff8000, "xvaddwod.h.bu.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74408000, 0xffff8000, "xvaddwod.w.hu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74410000, 0xffff8000, "xvaddwod.d.wu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74418000, 0xffff8000, "xvaddwod.q.du.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ac0000, 0xffff8000, "xvmaddwev.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ac8000, 0xffff8000, "xvmaddwev.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ad0000, 0xffff8000, "xvmaddwev.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ad8000, 0xffff8000, "xvmaddwev.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74bc0000, 0xffff8000, "xvmaddwev.h.bu.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74bc8000, 0xffff8000, "xvmaddwev.w.hu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74bd0000, 0xffff8000, "xvmaddwev.d.wu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74bd8000, 0xffff8000, "xvmaddwev.q.du.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74b40000, 0xffff8000, "xvmaddwev.h.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74b48000, 0xffff8000, "xvmaddwev.w.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74b50000, 0xffff8000, "xvmaddwev.d.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74b58000, 0xffff8000, "xvmaddwev.q.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ae0000, 0xffff8000, "xvmaddwod.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ae8000, 0xffff8000, "xvmaddwod.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74af0000, 0xffff8000, "xvmaddwod.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74af8000, 0xffff8000, "xvmaddwod.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74b60000, 0xffff8000, "xvmaddwod.h.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74b68000, 0xffff8000, "xvmaddwod.w.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74b70000, 0xffff8000, "xvmaddwod.d.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74b78000, 0xffff8000, "xvmaddwod.q.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74be0000, 0xffff8000, "xvmaddwod.h.bu.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74be8000, 0xffff8000, "xvmaddwod.w.hu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74bf0000, 0xffff8000, "xvmaddwod.d.wu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74bf8000, 0xffff8000, "xvmaddwod.q.du.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74900000, 0xffff8000, "xvmulwev.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74908000, 0xffff8000, "xvmulwev.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74910000, 0xffff8000, "xvmulwev.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74918000, 0xffff8000, "xvmulwev.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74980000, 0xffff8000, "xvmulwev.h.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74988000, 0xffff8000, "xvmulwev.w.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74990000, 0xffff8000, "xvmulwev.d.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74998000, 0xffff8000, "xvmulwev.q.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a00000, 0xffff8000, "xvmulwev.h.bu.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a08000, 0xffff8000, "xvmulwev.w.hu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a10000, 0xffff8000, "xvmulwev.d.wu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a18000, 0xffff8000, "xvmulwev.q.du.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74920000, 0xffff8000, "xvmulwod.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74928000, 0xffff8000, "xvmulwod.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74930000, 0xffff8000, "xvmulwod.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74938000, 0xffff8000, "xvmulwod.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x749a0000, 0xffff8000, "xvmulwod.h.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x749a8000, 0xffff8000, "xvmulwod.w.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x749b0000, 0xffff8000, "xvmulwod.d.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x749b8000, 0xffff8000, "xvmulwod.q.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a20000, 0xffff8000, "xvmulwod.h.bu.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a28000, 0xffff8000, "xvmulwod.w.hu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a30000, 0xffff8000, "xvmulwod.d.wu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a38000, 0xffff8000, "xvmulwod.q.du.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74200000, 0xffff8000, "xvsubwev.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74208000, 0xffff8000, "xvsubwev.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74210000, 0xffff8000, "xvsubwev.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74218000, 0xffff8000, "xvsubwev.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74300000, 0xffff8000, "xvsubwev.h.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74308000, 0xffff8000, "xvsubwev.w.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74310000, 0xffff8000, "xvsubwev.d.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74318000, 0xffff8000, "xvsubwev.q.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74240000, 0xffff8000, "xvsubwod.h.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74248000, 0xffff8000, "xvsubwod.w.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74250000, 0xffff8000, "xvsubwod.d.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74258000, 0xffff8000, "xvsubwod.q.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74340000, 0xffff8000, "xvsubwod.h.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74348000, 0xffff8000, "xvsubwod.w.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74350000, 0xffff8000, "xvsubwod.d.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74358000, 0xffff8000, "xvsubwod.q.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x745c0000, 0xffff8000, "xvadda.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x745c8000, 0xffff8000, "xvadda.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x745d0000, 0xffff8000, "xvadda.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x745d8000, 0xffff8000, "xvadda.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74600000, 0xffff8000, "xvabsd.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74608000, 0xffff8000, "xvabsd.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74610000, 0xffff8000, "xvabsd.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74618000, 0xffff8000, "xvabsd.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74620000, 0xffff8000, "xvabsd.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74628000, 0xffff8000, "xvabsd.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74630000, 0xffff8000, "xvabsd.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74638000, 0xffff8000, "xvabsd.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74640000, 0xffff8000, "xvavg.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74648000, 0xffff8000, "xvavg.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74650000, 0xffff8000, "xvavg.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74658000, 0xffff8000, "xvavg.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74660000, 0xffff8000, "xvavg.bu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74668000, 0xffff8000, "xvavg.hu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74670000, 0xffff8000, "xvavg.wu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74678000, 0xffff8000, "xvavg.du",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74680000, 0xffff8000, "xvavgr.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74688000, 0xffff8000, "xvavgr.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74690000, 0xffff8000, "xvavgr.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74698000, 0xffff8000, "xvavgr.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x746a0000, 0xffff8000, "xvavgr.bu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x746a8000, 0xffff8000, "xvavgr.hu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x746b0000, 0xffff8000, "xvavgr.wu",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x746b8000, 0xffff8000, "xvavgr.du",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74700000, 0xffff8000, "xvmax.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74708000, 0xffff8000, "xvmax.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74710000, 0xffff8000, "xvmax.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74718000, 0xffff8000, "xvmax.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74720000, 0xffff8000, "xvmin.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74728000, 0xffff8000, "xvmin.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74730000, 0xffff8000, "xvmin.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74738000, 0xffff8000, "xvmin.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74740000, 0xffff8000, "xvmax.bu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74748000, 0xffff8000, "xvmax.hu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74750000, 0xffff8000, "xvmax.wu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74758000, 0xffff8000, "xvmax.du",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74760000, 0xffff8000, "xvmin.bu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74768000, 0xffff8000, "xvmin.hu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74770000, 0xffff8000, "xvmin.wu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74778000, 0xffff8000, "xvmin.du",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74840000, 0xffff8000, "xvmul.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74848000, 0xffff8000, "xvmul.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74850000, 0xffff8000, "xvmul.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74858000, 0xffff8000, "xvmul.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74860000, 0xffff8000, "xvmuh.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74868000, 0xffff8000, "xvmuh.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74870000, 0xffff8000, "xvmuh.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74878000, 0xffff8000, "xvmuh.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74880000, 0xffff8000, "xvmuh.bu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74888000, 0xffff8000, "xvmuh.hu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74890000, 0xffff8000, "xvmuh.wu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74898000, 0xffff8000, "xvmuh.du",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a80000, 0xffff8000, "xvmadd.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a88000, 0xffff8000, "xvmadd.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a90000, 0xffff8000, "xvmadd.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74a98000, 0xffff8000, "xvmadd.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74aa0000, 0xffff8000, "xvmsub.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74aa8000, 0xffff8000, "xvmsub.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ab0000, 0xffff8000, "xvmsub.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ab8000, 0xffff8000, "xvmsub.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e00000, 0xffff8000, "xvdiv.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e08000, 0xffff8000, "xvdiv.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e10000, 0xffff8000, "xvdiv.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e18000, 0xffff8000, "xvdiv.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e20000, 0xffff8000, "xvmod.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e28000, 0xffff8000, "xvmod.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e30000, 0xffff8000, "xvmod.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e38000, 0xffff8000, "xvmod.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e40000, 0xffff8000, "xvdiv.bu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e48000, 0xffff8000, "xvdiv.hu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e50000, 0xffff8000, "xvdiv.wu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e58000, 0xffff8000, "xvdiv.du",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e60000, 0xffff8000, "xvmod.bu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e68000, 0xffff8000, "xvmod.hu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e70000, 0xffff8000, "xvmod.wu",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e78000, 0xffff8000, "xvmod.du",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e80000, 0xffff8000, "xvsll.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e88000, 0xffff8000, "xvsll.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e90000, 0xffff8000, "xvsll.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74e98000, 0xffff8000, "xvsll.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ea0000, 0xffff8000, "xvsrl.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ea8000, 0xffff8000, "xvsrl.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74eb0000, 0xffff8000, "xvsrl.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74eb8000, 0xffff8000, "xvsrl.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ec0000, 0xffff8000, "xvsra.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ec8000, 0xffff8000, "xvsra.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ed0000, 0xffff8000, "xvsra.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ed8000, 0xffff8000, "xvsra.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ee0000, 0xffff8000, "xvrotr.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ee8000, 0xffff8000, "xvrotr.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ef0000, 0xffff8000, "xvrotr.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ef8000, 0xffff8000, "xvrotr.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f00000, 0xffff8000, "xvsrlr.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f08000, 0xffff8000, "xvsrlr.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f10000, 0xffff8000, "xvsrlr.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f18000, 0xffff8000, "xvsrlr.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f20000, 0xffff8000, "xvsrar.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f28000, 0xffff8000, "xvsrar.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f30000, 0xffff8000, "xvsrar.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f38000, 0xffff8000, "xvsrar.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f48000, 0xffff8000, "xvsrln.b.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f50000, 0xffff8000, "xvsrln.h.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f58000, 0xffff8000, "xvsrln.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f68000, 0xffff8000, "xvsran.b.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f70000, 0xffff8000, "xvsran.h.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f78000, 0xffff8000, "xvsran.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f88000, 0xffff8000, "xvsrlrn.b.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f90000, 0xffff8000, "xvsrlrn.h.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74f98000, 0xffff8000, "xvsrlrn.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74fa8000, 0xffff8000, "xvsrarn.b.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74fb0000, 0xffff8000, "xvsrarn.h.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74fb8000, 0xffff8000, "xvsrarn.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74fc8000, 0xffff8000, "xvssrln.b.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74fd0000, 0xffff8000, "xvssrln.h.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74fd8000, 0xffff8000, "xvssrln.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74fe8000, 0xffff8000, "xvssran.b.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ff0000, 0xffff8000, "xvssran.h.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x74ff8000, 0xffff8000, "xvssran.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75008000, 0xffff8000, "xvssrlrn.b.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75010000, 0xffff8000, "xvssrlrn.h.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75018000, 0xffff8000, "xvssrlrn.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75028000, 0xffff8000, "xvssrarn.b.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75030000, 0xffff8000, "xvssrarn.h.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75038000, 0xffff8000, "xvssrarn.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75048000, 0xffff8000, "xvssrln.bu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75050000, 0xffff8000, "xvssrln.hu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75058000, 0xffff8000, "xvssrln.wu.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75068000, 0xffff8000, "xvssran.bu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75070000, 0xffff8000, "xvssran.hu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75078000, 0xffff8000, "xvssran.wu.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75088000, 0xffff8000, "xvssrlrn.bu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75090000, 0xffff8000, "xvssrlrn.hu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75098000, 0xffff8000, "xvssrlrn.wu.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750a8000, 0xffff8000, "xvssrarn.bu.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750b0000, 0xffff8000, "xvssrarn.hu.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750b8000, 0xffff8000, "xvssrarn.wu.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750c0000, 0xffff8000, "xvbitclr.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750c8000, 0xffff8000, "xvbitclr.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750d0000, 0xffff8000, "xvbitclr.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750d8000, 0xffff8000, "xvbitclr.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750e0000, 0xffff8000, "xvbitset.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750e8000, 0xffff8000, "xvbitset.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750f0000, 0xffff8000, "xvbitset.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x750f8000, 0xffff8000, "xvbitset.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75100000, 0xffff8000, "xvbitrev.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75108000, 0xffff8000, "xvbitrev.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75110000, 0xffff8000, "xvbitrev.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75118000, 0xffff8000, "xvbitrev.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75160000, 0xffff8000, "xvpackev.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75168000, 0xffff8000, "xvpackev.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75170000, 0xffff8000, "xvpackev.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75178000, 0xffff8000, "xvpackev.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75180000, 0xffff8000, "xvpackod.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75188000, 0xffff8000, "xvpackod.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75190000, 0xffff8000, "xvpackod.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75198000, 0xffff8000, "xvpackod.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751a0000, 0xffff8000, "xvilvl.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751a8000, 0xffff8000, "xvilvl.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751b0000, 0xffff8000, "xvilvl.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751b8000, 0xffff8000, "xvilvl.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751c0000, 0xffff8000, "xvilvh.b",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751c8000, 0xffff8000, "xvilvh.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751d0000, 0xffff8000, "xvilvh.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751d8000, 0xffff8000, "xvilvh.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751e0000, 0xffff8000, "xvpickev.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751e8000, 0xffff8000, "xvpickev.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751f0000, 0xffff8000, "xvpickev.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x751f8000, 0xffff8000, "xvpickev.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75200000, 0xffff8000, "xvpickod.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75208000, 0xffff8000, "xvpickod.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75210000, 0xffff8000, "xvpickod.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75218000, 0xffff8000, "xvpickod.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75220000, 0xffff8000, "xvreplve.b",	"x0:5,x5:5,r10:5",	0, 0, 0, 0},
  { 0x75228000, 0xffff8000, "xvreplve.h",	"x0:5,x5:5,r10:5",	0, 0, 0, 0},
  { 0x75230000, 0xffff8000, "xvreplve.w",	"x0:5,x5:5,r10:5",	0, 0, 0, 0},
  { 0x75238000, 0xffff8000, "xvreplve.d",	"x0:5,x5:5,r10:5",	0, 0, 0, 0},
  { 0x75260000, 0xffff8000, "xvand.v",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75268000, 0xffff8000, "xvor.v",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75270000, 0xffff8000, "xvxor.v",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75278000, 0xffff8000, "xvnor.v",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75280000, 0xffff8000, "xvandn.v",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75288000, 0xffff8000, "xvorn.v",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x752b0000, 0xffff8000, "xvfrstp.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x752b8000, 0xffff8000, "xvfrstp.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x752d0000, 0xffff8000, "xvadd.q",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x752d8000, 0xffff8000, "xvsub.q",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x752e0000, 0xffff8000, "xvsigncov.b",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x752e8000, 0xffff8000, "xvsigncov.h",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x752f0000, 0xffff8000, "xvsigncov.w",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x752f8000, 0xffff8000, "xvsigncov.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75308000, 0xffff8000, "xvfadd.s",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75310000, 0xffff8000, "xvfadd.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75328000, 0xffff8000, "xvfsub.s",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75330000, 0xffff8000, "xvfsub.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75388000, 0xffff8000, "xvfmul.s",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75390000, 0xffff8000, "xvfmul.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x753a8000, 0xffff8000, "xvfdiv.s",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x753b0000, 0xffff8000, "xvfdiv.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x753c8000, 0xffff8000, "xvfmax.s",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x753d0000, 0xffff8000, "xvfmax.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x753e8000, 0xffff8000, "xvfmin.s",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x753f0000, 0xffff8000, "xvfmin.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75408000, 0xffff8000, "xvfmaxa.s",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75410000, 0xffff8000, "xvfmaxa.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75428000, 0xffff8000, "xvfmina.s",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75430000, 0xffff8000, "xvfmina.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75460000, 0xffff8000, "xvfcvt.h.s",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75468000, 0xffff8000, "xvfcvt.s.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75480000, 0xffff8000, "xvffint.s.l",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x75498000, 0xffff8000, "xvftint.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x754a0000, 0xffff8000, "xvftintrm.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x754a8000, 0xffff8000, "xvftintrp.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x754b0000, 0xffff8000, "xvftintrz.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x754b8000, 0xffff8000, "xvftintrne.w.d",	"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x757a8000, 0xffff8000, "xvshuf.h",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x757b0000, 0xffff8000, "xvshuf.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x757b8000, 0xffff8000, "xvshuf.d",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x757d0000, 0xffff8000, "xvperm.w",		"x0:5,x5:5,x10:5",	0, 0, 0, 0 },
  { 0x76800000, 0xffff8000, "xvseqi.b",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76808000, 0xffff8000, "xvseqi.h",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76810000, 0xffff8000, "xvseqi.w",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76818000, 0xffff8000, "xvseqi.d",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76820000, 0xffff8000, "xvslei.b",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76828000, 0xffff8000, "xvslei.h",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76830000, 0xffff8000, "xvslei.w",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76838000, 0xffff8000, "xvslei.d",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76840000, 0xffff8000, "xvslei.bu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76848000, 0xffff8000, "xvslei.hu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76850000, 0xffff8000, "xvslei.wu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76858000, 0xffff8000, "xvslei.du",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76860000, 0xffff8000, "xvslti.b",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76868000, 0xffff8000, "xvslti.h",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76870000, 0xffff8000, "xvslti.w",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76878000, 0xffff8000, "xvslti.d",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76880000, 0xffff8000, "xvslti.bu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76888000, 0xffff8000, "xvslti.hu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76890000, 0xffff8000, "xvslti.wu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76898000, 0xffff8000, "xvslti.du",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768a0000, 0xffff8000, "xvaddi.bu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768a8000, 0xffff8000, "xvaddi.hu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768b0000, 0xffff8000, "xvaddi.wu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768b8000, 0xffff8000, "xvaddi.du",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768c0000, 0xffff8000, "xvsubi.bu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768c8000, 0xffff8000, "xvsubi.hu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768d0000, 0xffff8000, "xvsubi.wu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768d8000, 0xffff8000, "xvsubi.du",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768e0000, 0xffff8000, "xvbsll.v",		"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x768e8000, 0xffff8000, "xvbsrl.v",		"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76900000, 0xffff8000, "xvmaxi.b",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76908000, 0xffff8000, "xvmaxi.h",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76910000, 0xffff8000, "xvmaxi.w",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76918000, 0xffff8000, "xvmaxi.d",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76920000, 0xffff8000, "xvmini.b",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76928000, 0xffff8000, "xvmini.h",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76930000, 0xffff8000, "xvmini.w",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76938000, 0xffff8000, "xvmini.d",		"x0:5,x5:5,s10:5",	0, 0, 0, 0},
  { 0x76940000, 0xffff8000, "xvmaxi.bu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76948000, 0xffff8000, "xvmaxi.hu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76950000, 0xffff8000, "xvmaxi.wu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76958000, 0xffff8000, "xvmaxi.du",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76960000, 0xffff8000, "xvmini.bu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76968000, 0xffff8000, "xvmini.hu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76970000, 0xffff8000, "xvmini.wu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76978000, 0xffff8000, "xvmini.du",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x769a0000, 0xffff8000, "xvfrstpi.b",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x769a8000, 0xffff8000, "xvfrstpi.h",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x769c0000, 0xfffffc00, "xvclo.b",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c0400, 0xfffffc00, "xvclo.h",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c0800, 0xfffffc00, "xvclo.w",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c0c00, 0xfffffc00, "xvclo.d",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c1000, 0xfffffc00, "xvclz.b",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c1400, 0xfffffc00, "xvclz.h",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c1800, 0xfffffc00, "xvclz.w",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c1c00, 0xfffffc00, "xvclz.d",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c2000, 0xfffffc00, "xvpcnt.b",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c2400, 0xfffffc00, "xvpcnt.h",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c2800, 0xfffffc00, "xvpcnt.w",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c2c00, 0xfffffc00, "xvpcnt.d",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c3000, 0xfffffc00, "xvneg.b",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c3400, 0xfffffc00, "xvneg.h",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c3800, 0xfffffc00, "xvneg.w",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c3c00, 0xfffffc00, "xvneg.d",		"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c4000, 0xfffffc00, "xvmskltz.b",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c4400, 0xfffffc00, "xvmskltz.h",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c4800, 0xfffffc00, "xvmskltz.w",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c4c00, 0xfffffc00, "xvmskltz.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c5000, 0xfffffc00, "xvmskgez.b",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c6000, 0xfffffc00, "xvmsknz.b",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769c9800, 0xfffffc18, "xvseteqz.v",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769c9c00, 0xfffffc18, "xvsetnez.v",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769ca000, 0xfffffc18, "xvsetanyeqz.b",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769ca400, 0xfffffc18, "xvsetanyeqz.h",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769ca800, 0xfffffc18, "xvsetanyeqz.w",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769cac00, 0xfffffc18, "xvsetanyeqz.d",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769cb000, 0xfffffc18, "xvsetallnez.b",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769cb400, 0xfffffc18, "xvsetallnez.h",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769cb800, 0xfffffc18, "xvsetallnez.w",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769cbc00, 0xfffffc18, "xvsetallnez.d",	"c0:3,x5:5",		0, 0, 0, 0},
  { 0x769cc400, 0xfffffc00, "xvflogb.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769cc800, 0xfffffc00, "xvflogb.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769cd400, 0xfffffc00, "xvfclass.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769cd800, 0xfffffc00, "xvfclass.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ce400, 0xfffffc00, "xvfsqrt.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ce800, 0xfffffc00, "xvfsqrt.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769cf400, 0xfffffc00, "xvfrecip.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769cf800, 0xfffffc00, "xvfrecip.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d0400, 0xfffffc00, "xvfrsqrt.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d0800, 0xfffffc00, "xvfrsqrt.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d1400, 0xfffffc00, "xvfrecipe.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d1800, 0xfffffc00, "xvfrecipe.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d2400, 0xfffffc00, "xvfrsqrte.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d2800, 0xfffffc00, "xvfrsqrte.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d3400, 0xfffffc00, "xvfrint.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d3800, 0xfffffc00, "xvfrint.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d4400, 0xfffffc00, "xvfrintrm.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d4800, 0xfffffc00, "xvfrintrm.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d5400, 0xfffffc00, "xvfrintrp.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d5800, 0xfffffc00, "xvfrintrp.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d6400, 0xfffffc00, "xvfrintrz.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d6800, 0xfffffc00, "xvfrintrz.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d7400, 0xfffffc00, "xvfrintrne.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769d7800, 0xfffffc00, "xvfrintrne.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769de800, 0xfffffc00, "xvfcvtl.s.h",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769dec00, 0xfffffc00, "xvfcvth.s.h",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769df000, 0xfffffc00, "xvfcvtl.d.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769df400, 0xfffffc00, "xvfcvth.d.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e0000, 0xfffffc00, "xvffint.s.w",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e0400, 0xfffffc00, "xvffint.s.wu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e0800, 0xfffffc00, "xvffint.d.l",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e0c00, 0xfffffc00, "xvffint.d.lu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e1000, 0xfffffc00, "xvffintl.d.w",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e1400, 0xfffffc00, "xvffinth.d.w",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e3000, 0xfffffc00, "xvftint.w.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e3400, 0xfffffc00, "xvftint.l.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e3800, 0xfffffc00, "xvftintrm.w.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e3c00, 0xfffffc00, "xvftintrm.l.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e4000, 0xfffffc00, "xvftintrp.w.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e4400, 0xfffffc00, "xvftintrp.l.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e4800, 0xfffffc00, "xvftintrz.w.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e4c00, 0xfffffc00, "xvftintrz.l.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e5000, 0xfffffc00, "xvftintrne.w.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e5400, 0xfffffc00, "xvftintrne.l.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e5800, 0xfffffc00, "xvftint.wu.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e5c00, 0xfffffc00, "xvftint.lu.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e7000, 0xfffffc00, "xvftintrz.wu.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e7400, 0xfffffc00, "xvftintrz.lu.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e8000, 0xfffffc00, "xvftintl.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e8400, 0xfffffc00, "xvftinth.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e8800, 0xfffffc00, "xvftintrml.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e8c00, 0xfffffc00, "xvftintrmh.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e9000, 0xfffffc00, "xvftintrpl.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e9400, 0xfffffc00, "xvftintrph.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e9800, 0xfffffc00, "xvftintrzl.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769e9c00, 0xfffffc00, "xvftintrzh.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ea000, 0xfffffc00, "xvftintrnel.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ea400, 0xfffffc00, "xvftintrneh.l.s",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ee000, 0xfffffc00, "xvexth.h.b",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ee400, 0xfffffc00, "xvexth.w.h",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ee800, 0xfffffc00, "xvexth.d.w",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769eec00, 0xfffffc00, "xvexth.q.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ef000, 0xfffffc00, "xvexth.hu.bu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ef400, 0xfffffc00, "xvexth.wu.hu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769ef800, 0xfffffc00, "xvexth.du.wu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769efc00, 0xfffffc00, "xvexth.qu.du",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f0000, 0xfffffc00, "xvreplgr2vr.b",	"x0:5,r5:5",		0, 0, 0, 0},
  { 0x769f0400, 0xfffffc00, "xvreplgr2vr.h",	"x0:5,r5:5",		0, 0, 0, 0},
  { 0x769f0800, 0xfffffc00, "xvreplgr2vr.w",	"x0:5,r5:5",		0, 0, 0, 0},
  { 0x769f0c00, 0xfffffc00, "xvreplgr2vr.d",	"x0:5,r5:5",		0, 0, 0, 0},
  { 0x769f1000, 0xfffffc00, "vext2xv.h.b",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f1400, 0xfffffc00, "vext2xv.w.b",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f1800, 0xfffffc00, "vext2xv.d.b",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f1c00, 0xfffffc00, "vext2xv.w.h",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f2000, 0xfffffc00, "vext2xv.d.h",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f2400, 0xfffffc00, "vext2xv.d.w",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f2800, 0xfffffc00, "vext2xv.hu.bu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f2c00, 0xfffffc00, "vext2xv.wu.bu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f3000, 0xfffffc00, "vext2xv.du.bu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f3400, 0xfffffc00, "vext2xv.wu.hu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f3800, 0xfffffc00, "vext2xv.du.hu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f3c00, 0xfffffc00, "vext2xv.du.wu",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x769f8000, 0xffff8000, "xvhseli.d",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76a02000, 0xffffe000, "xvrotri.b",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x76a04000, 0xffffc000, "xvrotri.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x76a08000, 0xffff8000, "xvrotri.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76a10000, 0xffff0000, "xvrotri.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x76a42000, 0xffffe000, "xvsrlri.b",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x76a44000, 0xffffc000, "xvsrlri.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x76a48000, 0xffff8000, "xvsrlri.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76a50000, 0xffff0000, "xvsrlri.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x76a82000, 0xffffe000, "xvsrari.b",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x76a84000, 0xffffc000, "xvsrari.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x76a88000, 0xffff8000, "xvsrari.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x76a90000, 0xffff0000, "xvsrari.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x76ebc000, 0xffffe000, "xvinsgr2vr.w",	"x0:5,r5:5,u10:3",	0, 0, 0, 0},
  { 0x76ebe000, 0xfffff000, "xvinsgr2vr.d",	"x0:5,r5:5,u10:2",	0, 0, 0, 0},
  { 0x76efc000, 0xffffe000, "xvpickve2gr.w",	"r0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x76efe000, 0xfffff000, "xvpickve2gr.d",	"r0:5,x5:5,u10:2",	0, 0, 0, 0},
  { 0x76f3c000, 0xffffe000, "xvpickve2gr.wu",	"r0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x76f3e000, 0xfffff000, "xvpickve2gr.du",	"r0:5,x5:5,u10:2",	0, 0, 0, 0},
  { 0x76f78000, 0xffffc000, "xvrepl128vei.b",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x76f7c000, 0xffffe000, "xvrepl128vei.h",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x76f7e000, 0xfffff000, "xvrepl128vei.w",	"x0:5,x5:5,u10:2",	0, 0, 0, 0},
  { 0x76f7f000, 0xfffff800, "xvrepl128vei.d",	"x0:5,x5:5,u10:1",	0, 0, 0, 0},
  { 0x76ffc000, 0xffffe000, "xvinsve0.w",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x76ffe000, 0xfffff000, "xvinsve0.d",	"x0:5,x5:5,u10:2",	0, 0, 0, 0},
  { 0x7703c000, 0xffffe000, "xvpickve.w",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x7703e000, 0xfffff000, "xvpickve.d",	"x0:5,x5:5,u10:2",	0, 0, 0, 0},
  { 0x77070000, 0xfffffc00, "xvreplve0.b",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x77078000, 0xfffffc00, "xvreplve0.h",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x7707c000, 0xfffffc00, "xvreplve0.w",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x7707e000, 0xfffffc00, "xvreplve0.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x7707f000, 0xfffffc00, "xvreplve0.q",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x77082000, 0xffffe000, "xvsllwil.h.b",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x77084000, 0xffffc000, "xvsllwil.w.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77088000, 0xffff8000, "xvsllwil.d.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77090000, 0xfffffc00, "xvextl.q.d",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x770c2000, 0xffffe000, "xvsllwil.hu.bu",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x770c4000, 0xffffc000, "xvsllwil.wu.hu",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x770c8000, 0xffff8000, "xvsllwil.du.wu",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x770d0000, 0xfffffc00, "xvextl.qu.du",	"x0:5,x5:5",		0, 0, 0, 0},
  { 0x77102000, 0xffffe000, "xvbitclri.b",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x77104000, 0xffffc000, "xvbitclri.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77108000, 0xffff8000, "xvbitclri.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77110000, 0xffff0000, "xvbitclri.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77142000, 0xffffe000, "xvbitseti.b",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x77144000, 0xffffc000, "xvbitseti.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77148000, 0xffff8000, "xvbitseti.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77150000, 0xffff0000, "xvbitseti.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77182000, 0xffffe000, "xvbitrevi.b",	"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x77184000, 0xffffc000, "xvbitrevi.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77188000, 0xffff8000, "xvbitrevi.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77190000, 0xffff0000, "xvbitrevi.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77242000, 0xffffe000, "xvsat.b",		"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x77244000, 0xffffc000, "xvsat.h",		"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77248000, 0xffff8000, "xvsat.w",		"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77250000, 0xffff0000, "xvsat.d",		"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77282000, 0xffffe000, "xvsat.bu",		"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x77284000, 0xffffc000, "xvsat.hu",		"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77288000, 0xffff8000, "xvsat.wu",		"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77290000, 0xffff0000, "xvsat.du",		"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x772c2000, 0xffffe000, "xvslli.b",		"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x772c4000, 0xffffc000, "xvslli.h",		"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x772c8000, 0xffff8000, "xvslli.w",		"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x772d0000, 0xffff0000, "xvslli.d",		"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77302000, 0xffffe000, "xvsrli.b",		"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x77304000, 0xffffc000, "xvsrli.h",		"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77308000, 0xffff8000, "xvsrli.w",		"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77310000, 0xffff0000, "xvsrli.d",		"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77342000, 0xffffe000, "xvsrai.b",		"x0:5,x5:5,u10:3",	0, 0, 0, 0},
  { 0x77344000, 0xffffc000, "xvsrai.h",		"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77348000, 0xffff8000, "xvsrai.w",		"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77350000, 0xffff0000, "xvsrai.d",		"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77404000, 0xffffc000, "xvsrlni.b.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77408000, 0xffff8000, "xvsrlni.h.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77410000, 0xffff0000, "xvsrlni.w.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77420000, 0xfffe0000, "xvsrlni.d.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x77444000, 0xffffc000, "xvsrlrni.b.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77448000, 0xffff8000, "xvsrlrni.h.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77450000, 0xffff0000, "xvsrlrni.w.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77460000, 0xfffe0000, "xvsrlrni.d.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x77484000, 0xffffc000, "xvssrlni.b.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77488000, 0xffff8000, "xvssrlni.h.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77490000, 0xffff0000, "xvssrlni.w.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x774a0000, 0xfffe0000, "xvssrlni.d.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x774c4000, 0xffffc000, "xvssrlni.bu.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x774c8000, 0xffff8000, "xvssrlni.hu.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x774d0000, 0xffff0000, "xvssrlni.wu.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x774e0000, 0xfffe0000, "xvssrlni.du.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x77504000, 0xffffc000, "xvssrlrni.b.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77508000, 0xffff8000, "xvssrlrni.h.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77510000, 0xffff0000, "xvssrlrni.w.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77520000, 0xfffe0000, "xvssrlrni.d.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x77544000, 0xffffc000, "xvssrlrni.bu.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77548000, 0xffff8000, "xvssrlrni.hu.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77550000, 0xffff0000, "xvssrlrni.wu.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77560000, 0xfffe0000, "xvssrlrni.du.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x77584000, 0xffffc000, "xvsrani.b.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77588000, 0xffff8000, "xvsrani.h.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77590000, 0xffff0000, "xvsrani.w.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x775a0000, 0xfffe0000, "xvsrani.d.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x775c4000, 0xffffc000, "xvsrarni.b.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x775c8000, 0xffff8000, "xvsrarni.h.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x775d0000, 0xffff0000, "xvsrarni.w.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x775e0000, 0xfffe0000, "xvsrarni.d.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x77604000, 0xffffc000, "xvssrani.b.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77608000, 0xffff8000, "xvssrani.h.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77610000, 0xffff0000, "xvssrani.w.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77620000, 0xfffe0000, "xvssrani.d.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x77644000, 0xffffc000, "xvssrani.bu.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77648000, 0xffff8000, "xvssrani.hu.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77650000, 0xffff0000, "xvssrani.wu.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x77660000, 0xfffe0000, "xvssrani.du.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x77684000, 0xffffc000, "xvssrarni.b.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x77688000, 0xffff8000, "xvssrarni.h.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x77690000, 0xffff0000, "xvssrarni.w.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x776a0000, 0xfffe0000, "xvssrarni.d.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x776c4000, 0xffffc000, "xvssrarni.bu.h",	"x0:5,x5:5,u10:4",	0, 0, 0, 0},
  { 0x776c8000, 0xffff8000, "xvssrarni.hu.w",	"x0:5,x5:5,u10:5",	0, 0, 0, 0},
  { 0x776d0000, 0xffff0000, "xvssrarni.wu.d",	"x0:5,x5:5,u10:6",	0, 0, 0, 0},
  { 0x776e0000, 0xfffe0000, "xvssrarni.du.q",	"x0:5,x5:5,u10:7",	0, 0, 0, 0},
  { 0x77800000, 0xfffc0000, "xvextrins.d",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77840000, 0xfffc0000, "xvextrins.w",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77880000, 0xfffc0000, "xvextrins.h",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x778c0000, 0xfffc0000, "xvextrins.b",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77900000, 0xfffc0000, "xvshuf4i.b",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77940000, 0xfffc0000, "xvshuf4i.h",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77980000, 0xfffc0000, "xvshuf4i.w",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x779c0000, 0xfffc0000, "xvshuf4i.d",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77c40000, 0xfffc0000, "xvbitseli.b",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77d00000, 0xfffc0000, "xvandi.b",		"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77d40000, 0xfffc0000, "xvori.b",		"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77d80000, 0xfffc0000, "xvxori.b",		"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77dc0000, 0xfffc0000, "xvnori.b",		"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0,		0,	    "xvrepli.b",	"x,s0:10",		"xvldi %1,(%2)&0x3ff",		0, 0, 0},
  { 0,		0,	    "xvrepli.d",	"x,s0:10",		"xvldi %1,((%2)&0x3ff)|0xc00",	0, 0, 0},
  { 0,		0,	    "xvrepli.h",	"x,s0:10",		"xvldi %1,((%2)&0x3ff)|0x400",	0, 0, 0},
  { 0,		0,	    "xvrepli.w",	"x,s0:10",		"xvldi %1,((%2)&0x3ff)|0x800",	0, 0, 0},
  { 0x77e00000, 0xfffc0000, "xvldi",		"x0:5,s5:13",		0, 0, 0, 0},
  { 0x77e40000, 0xfffc0000, "xvpermi.w",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77e80000, 0xfffc0000, "xvpermi.d",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0x77ec0000, 0xfffc0000, "xvpermi.q",	"x0:5,x5:5,u10:8",	0, 0, 0, 0},
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list. */
};

static struct loongarch_opcode loongarch_lvz_opcodes[] =
{
  /* match,    mask,	   name,	format,			macro, include, exclude, pinfo.  */
  {0x05000000, 0xff0003e0, "gcsrrd",	"r0:5,u10:14",		0, 0, 0, 0},
  {0x05000020, 0xff0003e0, "gcsrwr",	"r0:5,u10:14",		0, 0, 0, 0},
  {0x05000000, 0xff000000, "gcsrxchg",	"r0:5,r5:5,u10:14",	0, 0, 0, 0},
  {0x06482401, 0xffffffff, "gtlbflush",	"",			0, 0, 0, 0},
  {0x002b8000, 0xffff8000, "hvcl",	"u0:15",		0, 0, 0, 0},
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

static struct loongarch_opcode loongarch_lbt_opcodes[] =
{
  /* match,    mask,	   name,		format,			macro, include, exclude, pinfo.  */
  {0x00000800, 0xfffffc1c, "movgr2scr",		"cr0:2,r5:5",		0, 0, 0, 0},
  {0x00000c00, 0xffffff80, "movscr2gr",		"r0:5,cr5:2",		0, 0, 0, 0},
  {0x48000200, 0xfc0003e0, "jiscr0",		"s0:5|10:16<<2",	0, 0, 0, 0},
  {0x48000300, 0xfc0003e0, "jiscr1",		"s0:5|10:16<<2",	0, 0, 0, 0},
  {0x00290000, 0xffff8000, "addu12i.w",		"r0:5,r5:5,s10:5",	0, 0, 0, 0},
  {0x00298000, 0xffff8000, "addu12i.d",		"r0:5,r5:5,s10:5",	0, 0, 0, 0},
  {0x00300000, 0xffff8000, "adc.b",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00308000, 0xffff8000, "adc.h",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00310000, 0xffff8000, "adc.w",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00318000, 0xffff8000, "adc.d",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00320000, 0xffff8000, "sbc.b",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00328000, 0xffff8000, "sbc.h",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00330000, 0xffff8000, "sbc.w",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00338000, 0xffff8000, "sbc.d",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x001a0000, 0xffff8000, "rotr.b",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x001a8000, 0xffff8000, "rotr.h",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x004c2000, 0xffffe000, "rotri.b",		"r0:5,r5:5,u10:3",	0, 0, 0, 0},
  {0x004c4000, 0xffffc000, "rotri.h",		"r0:5,r5:5,u10:4",	0, 0, 0, 0},
  {0x00340000, 0xffff8000, "rcr.b",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00348000, 0xffff8000, "rcr.h",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00350000, 0xffff8000, "rcr.w",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00358000, 0xffff8000, "rcr.d",		"r0:5,r5:5,r10:5",	0, 0, 0, 0},
  {0x00502000, 0xffffe000, "rcri.b",		"r0:5,r5:5,u10:3",	0, 0, 0, 0},
  {0x00504000, 0xffffc000, "rcri.h",		"r0:5,r5:5,u10:4",	0, 0, 0, 0},
  {0x00508000, 0xffff8000, "rcri.w",		"r0:5,r5:5,u10:5",	0, 0, 0, 0},
  {0x00510000, 0xffff0000, "rcri.d",		"r0:5,r5:5,u10:6",	0, 0, 0, 0},
  {0x0114e400, 0xfffffc00, "fcvt.ud.d",		"f0:5,f5:5",		0, 0, 0, 0},
  {0x0114e000, 0xfffffc00, "fcvt.ld.d",		"f0:5,f5:5",		0, 0, 0, 0},
  {0x01150000, 0xffff8000, "fcvt.d.ld",		"f0:5,f5:5,f10:5",	0, 0, 0, 0},
  {0x2e800000, 0xffc00000, "ldl.d",		"r0:5,r5:5,so10:12",	0, 0, 0, 0},
  {0x2e000000, 0xffc00000, "ldl.w",		"r0:5,r5:5,so10:12",	0, 0, 0, 0},
  {0x2e400000, 0xffc00000, "ldr.w",		"r0:5,r5:5,so10:12",	0, 0, 0, 0},
  {0x2ec00000, 0xffc00000, "ldr.d",		"r0:5,r5:5,so10:12",	0, 0, 0, 0},
  {0x2f000000, 0xffc00000, "stl.w",		"r0:5,r5:5,so10:12",	0, 0, 0, 0},
  {0x2f800000, 0xffc00000, "stl.d",		"r0:5,r5:5,so10:12",	0, 0, 0, 0},
  {0x2f400000, 0xffc00000, "str.w",		"r0:5,r5:5,so10:12",	0, 0, 0, 0},
  {0x2fc00000, 0xffc00000, "str.d",		"r0:5,r5:5,so10:12",	0, 0, 0, 0},
  {0x003f000c, 0xffff801f, "x86adc.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f000d, 0xffff801f, "x86adc.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f000e, 0xffff801f, "x86adc.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f000f, 0xffff801f, "x86adc.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0004, 0xffff801f, "x86add.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0005, 0xffff801f, "x86add.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0006, 0xffff801f, "x86add.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0007, 0xffff801f, "x86add.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0000, 0xffff801f, "x86add.wu",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0001, 0xffff801f, "x86add.du",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x00008000, 0xfffffc1f, "x86inc.b",		"r5:5",			0, 0, 0, 0},
  {0x00008001, 0xfffffc1f, "x86inc.h",		"r5:5",			0, 0, 0, 0},
  {0x00008002, 0xfffffc1f, "x86inc.w",		"r5:5",			0, 0, 0, 0},
  {0x00008003, 0xfffffc1f, "x86inc.d",		"r5:5",			0, 0, 0, 0},
  {0x003f0010, 0xffff801f, "x86sbc.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0011, 0xffff801f, "x86sbc.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0012, 0xffff801f, "x86sbc.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0013, 0xffff801f, "x86sbc.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0008, 0xffff801f, "x86sub.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0009, 0xffff801f, "x86sub.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f000a, 0xffff801f, "x86sub.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f000b, 0xffff801f, "x86sub.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0002, 0xffff801f, "x86sub.wu",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0003, 0xffff801f, "x86sub.du",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x00008004, 0xfffffc1f, "x86dec.b",		"r5:5",			0, 0, 0, 0},
  {0x00008005, 0xfffffc1f, "x86dec.h",		"r5:5",			0, 0, 0, 0},
  {0x00008006, 0xfffffc1f, "x86dec.w",		"r5:5",			0, 0, 0, 0},
  {0x00008007, 0xfffffc1f, "x86dec.d",		"r5:5",			0, 0, 0, 0},
  {0x003f8010, 0xffff801f, "x86and.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8011, 0xffff801f, "x86and.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8012, 0xffff801f, "x86and.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8013, 0xffff801f, "x86and.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8014, 0xffff801f, "x86or.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8015, 0xffff801f, "x86or.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8016, 0xffff801f, "x86or.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8017, 0xffff801f, "x86or.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8018, 0xffff801f, "x86xor.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8019, 0xffff801f, "x86xor.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f801a, 0xffff801f, "x86xor.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f801b, 0xffff801f, "x86xor.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003e8000, 0xffff801f, "x86mul.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003e8001, 0xffff801f, "x86mul.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003e8002, 0xffff801f, "x86mul.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003e8003, 0xffff801f, "x86mul.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003e8004, 0xffff801f, "x86mul.bu",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003e8005, 0xffff801f, "x86mul.hu",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003e8006, 0xffff801f, "x86mul.wu",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003e8007, 0xffff801f, "x86mul.du",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f800c, 0xffff801f, "x86rcl.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f800d, 0xffff801f, "x86rcl.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f800e, 0xffff801f, "x86rcl.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f800f, 0xffff801f, "x86rcl.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x00542018, 0xffffe01f, "x86rcli.b",		"r5:5,u10:3",		0, 0, 0, 0},
  {0x00544019, 0xffffc01f, "x86rcli.h",		"r5:5,u10:4",		0, 0, 0, 0},
  {0x0054801a, 0xffff801f, "x86rcli.w",		"r5:5,u10:5",		0, 0, 0, 0},
  {0x0055001b, 0xffff001f, "x86rcli.d",		"r5:5,u10:6",		0, 0, 0, 0},
  {0x003f8008, 0xffff801f, "x86rcr.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8009, 0xffff801f, "x86rcr.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f800a, 0xffff801f, "x86rcr.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f800b, 0xffff801f, "x86rcr.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x00542010, 0xffffe01f, "x86rcri.b",		"r5:5,u10:3",		0, 0, 0, 0},
  {0x00544011, 0xffffc01f, "x86rcri.h",		"r5:5,u10:4",		0, 0, 0, 0},
  {0x00548012, 0xffff801f, "x86rcri.w",		"r5:5,u10:5",		0, 0, 0, 0},
  {0x00550013, 0xffff001f, "x86rcri.d",		"r5:5,u10:6",		0, 0, 0, 0},
  {0x003f8004, 0xffff801f, "x86rotl.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8005, 0xffff801f, "x86rotl.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8006, 0xffff801f, "x86rotl.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8007, 0xffff801f, "x86rotl.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x00542014, 0xffffe01f, "x86rotli.b",	"r5:5,u10:3",		0, 0, 0, 0},
  {0x00544015, 0xffffc01f, "x86rotli.h",	"r5:5,u10:4",		0, 0, 0, 0},
  {0x00548016, 0xffff801f, "x86rotli.w",	"r5:5,u10:5",		0, 0, 0, 0},
  {0x00550017, 0xffff001f, "x86rotli.d",	"r5:5,u10:6",		0, 0, 0, 0},
  {0x003f8000, 0xffff801f, "x86rotr.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8001, 0xffff801f, "x86rotr.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8002, 0xffff801f, "x86rotr.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f8003, 0xffff801f, "x86rotr.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x0054200c, 0xffffe01f, "x86rotri.b",	"r5:5,u10:3",		0, 0, 0, 0},
  {0x0054400d, 0xffffc01f, "x86rotri.h",	"r5:5,u10:4",		0, 0, 0, 0},
  {0x0054800e, 0xffff801f, "x86rotri.w",	"r5:5,u10:5",		0, 0, 0, 0},
  {0x0055000f, 0xffff001f, "x86rotri.d",	"r5:5,u10:6",		0, 0, 0, 0},
  {0x003f0014, 0xffff801f, "x86sll.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0015, 0xffff801f, "x86sll.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0016, 0xffff801f, "x86sll.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0017, 0xffff801f, "x86sll.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x00542000, 0xffffe01f, "x86slli.b",		"r5:5,u10:3",		0, 0, 0, 0},
  {0x00544001, 0xffffc01f, "x86slli.h",		"r5:5,u10:4",		0, 0, 0, 0},
  {0x00548002, 0xffff801f, "x86slli.w",		"r5:5,u10:5",		0, 0, 0, 0},
  {0x00550003, 0xffff001f, "x86slli.d",		"r5:5,u10:6",		0, 0, 0, 0},
  {0x003f0018, 0xffff801f, "x86srl.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f0019, 0xffff801f, "x86srl.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f001a, 0xffff801f, "x86srl.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f001b, 0xffff801f, "x86srl.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x00542004, 0xffffe01f, "x86srli.b",		"r5:5,u10:3",		0, 0, 0, 0},
  {0x00544005, 0xffffc01f, "x86srli.h",		"r5:5,u10:4",		0, 0, 0, 0},
  {0x00548006, 0xffff801f, "x86srli.w",		"r5:5,u10:5",		0, 0, 0, 0},
  {0x00550007, 0xffff001f, "x86srli.d",		"r5:5,u10:6",		0, 0, 0, 0},
  {0x003f001c, 0xffff801f, "x86sra.b",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f001d, 0xffff801f, "x86sra.h",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f001e, 0xffff801f, "x86sra.w",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x003f001f, 0xffff801f, "x86sra.d",		"r5:5,r10:5",		0, 0, 0, 0},
  {0x00542008, 0xffffe01f, "x86srai.b",		"r5:5,u10:3",		0, 0, 0, 0},
  {0x00544009, 0xffffc01f, "x86srai.h",		"r5:5,u10:4",		0, 0, 0, 0},
  {0x0054800a, 0xffff801f, "x86srai.w",		"r5:5,u10:5",		0, 0, 0, 0},
  {0x0055000b, 0xffff001f, "x86srai.d",		"r5:5,u10:6",		0, 0, 0, 0},
  {0x00368000, 0xffffc3e0, "setx86j",		"r0:5,u10:4",		0, 0, 0, 0},
  {0x00007800, 0xfffffc00, "setx86loope",	"r0:5,r5:5",		0, 0, 0, 0},
  {0x00007c00, 0xfffffc00, "setx86loopne",	"r0:5,r5:5",		0, 0, 0, 0},
  {0x005c0000, 0xfffc03e0, "x86mfflag",		"r0:5,u10:8",		0, 0, 0, 0},
  {0x005c0020, 0xfffc03e0, "x86mtflag",		"r0:5,u10:8",		0, 0, 0, 0},
  {0x00007400, 0xffffffe0, "x86mftop",		"r0:5",			0, 0, 0, 0},
  {0x00007000, 0xffffff1f, "x86mttop",		"u5:3",			0, 0, 0, 0},
  {0x00008009, 0xffffffff, "x86inctop",		"",			0, 0, 0, 0},
  {0x00008029, 0xffffffff, "x86dectop",		"",			0, 0, 0, 0},
  {0x00008008, 0xffffffff, "x86settm",		"",			0, 0, 0, 0},
  {0x00008028, 0xffffffff, "x86clrtm",		"",			0, 0, 0, 0},
  {0x00580000, 0xfffc0000, "x86settag",		"r0:5,u5:5,u10:8",	0, 0, 0, 0},
  {0x00370010, 0xffff8010, "armadd.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x00378010, 0xffff8010, "armsub.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x00380010, 0xffff8010, "armadc.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x00388010, 0xffff8010, "armsbc.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x00390010, 0xffff8010, "armand.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x00398010, 0xffff8010, "armor.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x003a0010, 0xffff8010, "armxor.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x003fc01c, 0xffffc01f, "armnot.w",		"r5:5,u10:4",		0, 0, 0, 0},
  {0x003a8010, 0xffff8010, "armsll.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x003b0010, 0xffff8010, "armsrl.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x003b8010, 0xffff8010, "armsra.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x003c0010, 0xffff8010, "armrotr.w",		"r5:5,r10:5,u0:4",	0, 0, 0, 0},
  {0x003c8010, 0xffff8010, "armslli.w",		"r5:5,u10:5,u0:4",	0, 0, 0, 0},
  {0x003d0010, 0xffff8010, "armsrli.w",		"r5:5,u10:5,u0:4",	0, 0, 0, 0},
  {0x003d8010, 0xffff8010, "armsrai.w",		"r5:5,u10:5,u0:4",	0, 0, 0, 0},
  {0x003e0010, 0xffff8010, "armrotri.w",	"r5:5,u10:5,u0:4",	0, 0, 0, 0},
  {0x003fc01f, 0xffffc01f, "armrrx.w",		"r5:5,u10:4",		0, 0, 0, 0},
  {0x00364000, 0xffffc000, "armmove",		"r0:5,r5:5,u10:4",	0, 0, 0, 0},
  {0x003fc01d, 0xffffc01f, "armmov.w",		"r5:5,u10:4",		0, 0, 0, 0},
  {0x003fc01e, 0xffffc01f, "armmov.d",		"r5:5,u10:4",		0, 0, 0, 0},
  {0x005c0040, 0xfffc03e0, "armmfflag",		"r0:5,u10:8",		0, 0, 0, 0},
  {0x005c0060, 0xfffc03e0, "armmtflag",		"r0:5,u10:8",		0, 0, 0, 0},
  {0x0036c000, 0xffffc3e0, "setarmj",		"r0:5,u10:4",		0, 0, 0, 0},
  { 0, 0, 0, 0, 0, 0, 0, 0 } /* Terminate the list.  */
};

struct loongarch_ase loongarch_ASEs[] =
{
  { &LARCH_opts.ase_ilp32, loongarch_macro_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_ilp32, loongarch_alias_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_ilp32, loongarch_imm_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_ilp32, loongarch_privilege_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_ilp32, loongarch_load_store_opcodes,	0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_ilp32, loongarch_fix_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_ilp32, loongarch_jmp_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_sf, loongarch_float_jmp_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_sf, loongarch_single_float_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_df, loongarch_double_float_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_sf, loongarch_4opt_single_float_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_df, loongarch_4opt_double_float_opcodes,		0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_sf, loongarch_single_float_load_store_opcodes,	0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_df, loongarch_double_float_load_store_opcodes,	0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_lsx,  loongarch_lsx_opcodes,  0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_lasx, loongarch_lasx_opcodes, 0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_lvz,  loongarch_lvz_opcodes,  0, 0, { 0 }, 0, 0 },
  { &LARCH_opts.ase_lbt,  loongarch_lbt_opcodes,  0, 0, { 0 }, 0, 0 },
  { 0, 0, 0, 0, { 0 }, 0, 0 },
};
