/* Copyright (C) 2017-2025 Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#include "amd64.h"
#include "gdbsupport/x86-xstate.h"
#include "gdbsupport/osabi.h"
#include <stdlib.h>

#include "../features/i386/64bit-avx.c"
#include "../features/i386/64bit-avx512.c"
#include "../features/i386/64bit-core.c"
#include "../features/i386/64bit-linux.c"
#include "../features/i386/64bit-segments.c"
#include "../features/i386/64bit-sse.c"
#include "../features/i386/pkeys.c"

#include "../features/i386/64bit-ssp.c"
#include "../features/i386/32bit-ssp.c"
#include "../features/i386/x32-core.c"

/* See arch/amd64.h.  */

target_desc_up
amd64_create_target_description (uint64_t xstate_bv, bool is_x32,
				 bool is_linux, bool segments)
{
  target_desc_up tdesc = allocate_target_description ();

#ifndef IN_PROCESS_AGENT
  set_tdesc_architecture (tdesc.get (),
			  is_x32 ? "i386:x64-32" : "i386:x86-64");

  if (is_linux)
    set_tdesc_osabi (tdesc.get (), GDB_OSABI_LINUX);
#endif

  long regnum = 0;

  if (is_x32)
    regnum = create_feature_i386_x32_core (tdesc.get (), regnum);
  else
    regnum = create_feature_i386_64bit_core (tdesc.get (), regnum);

  regnum = create_feature_i386_64bit_sse (tdesc.get (), regnum);
  if (is_linux)
    regnum = create_feature_i386_64bit_linux (tdesc.get (), regnum);
  if (segments)
    regnum = create_feature_i386_64bit_segments (tdesc.get (), regnum);

  if (xstate_bv & X86_XSTATE_AVX)
    regnum = create_feature_i386_64bit_avx (tdesc.get (), regnum);

  if (xstate_bv & X86_XSTATE_AVX512)
    regnum = create_feature_i386_64bit_avx512 (tdesc.get (), regnum);

  if (xstate_bv & X86_XSTATE_PKRU)
    regnum = create_feature_i386_pkeys (tdesc.get (), regnum);

  if (xstate_bv & X86_XSTATE_CET_U)
    {
      if (!is_x32)
	regnum = create_feature_i386_64bit_ssp (tdesc.get (), regnum);
      else
	regnum = create_feature_i386_32bit_ssp (tdesc.get (), regnum);
    }

  return tdesc;
}
